#' Trelliscope Visualisation of Overimputation Diagnostics
#' @description Generates a Trelliscope display for overimputation diagnostics across all variables.
#' @param obj An object of class 'overimp' containing imputed datasets and parameters.
#' @param m A single positive integer specifying the number of imputed datasets to plot. It should be smaller than the total number of imputed datasets in the object. Default is NULL (
#' plot all).
#' @param imp_idx A vector of integers specifying the indices of imputed datasets to plot. Default is NULL (plot all).
#' @param integerAsFactor A logical indicating whether integer variables should be treated as factors. Default is FALSE (treated as numeric).
#' @param title A string specifying the title of the plot. Default is "auto" (automatic title). If NULL, no title is shown.
#' @param subtitle A string specifying the subtitle of the plot. Default is "auto" (automatic subtitle). If NULL, no subtitle is shown.
#' @param num_plot A character string specifying the type of plot for numeric variables. Options are "cv" (cross-validation), "ridge", or "density". Default is "cv".
#' @param fac_plot A character string specifying the type of plot for categorical variables. Options are "cv" (cross-validation), "bar", or "dodge". Default is "cv".
#' @param train_color_pal A vector of colors for the training data. If NULL, default colors will be used.
#' @param test_color_pal A vector of colors for the test data. If NULL, default colors will be used.
#' @param stack_y A logical indicating whether to stack y-values in the plots. Default is FALSE.
#' @param diag_color A color specification for the diagonal line in the plots. Default is NULL.
#' @param seed An integer seed for reproducibility. Default is 2025.
#' @param nrow Number of rows in the Trelliscope display. Default is 2.
#' @param ncol Number of columns in the Trelliscope display. Default is 4.
#' @param path Optional path to save the Trelliscope display. If NULL, the display will not be saved to disk.
#' @param verbose A logical value indicating whether to print extra information. Default is FALSE.
#' @param ... Additional arguments to customize the plots, such as point_size, xlim, ylim.
#' @return A Trelliscope display object visualising overimputation diagnostics for all variables.
#' @export
#' @examples
#' obj <- overimp(data = nhanes3, m = 3, p = 0.2, test_ratio = 0, method = "mixgb")
#' trellis_vismi_overimp(obj = obj, stack_y = TRUE)
trellis_vismi_overimp <- function(obj, m = NULL, imp_idx = NULL, integerAsFactor = FALSE, title = "auto", subtitle = "auto", num_plot = "cv", fac_plot = "cv", train_color_pal = NULL, test_color_pal = NULL, stack_y = FALSE, diag_color = "white", seed = 2025, nrow = 2, ncol = 4, path = NULL, verbose = FALSE, ...) {
  Variable <- obj$params$Names

  Types <- obj$params$Types
  Types[Types == "integer"] <- if (isTRUE(integerAsFactor)) "factor" else "numeric"

  users_params <- list(...)
  params <- modifyList(.vismi_overimp_params(), users_params)

  plot_map <- list(
    numeric = list(cv = overimp1D_cv_num, ridge = overimp1D_ridge, density = overimp1D_density),
    factor  = list(cv = overimp1D_cv_fac, bar = overimp1D_bar, dodge = overimp1D_dodge)
  )


  default_title <- "Masked true vs multiply-imputed values:"

  # Create a "Control Table" for Trelliscope. This table has one row per variable
  all_vars_df <- tibble(Variable = Variable) |>
    group_by(Variable) |>
    mutate(panel = purrr::map(Variable, function(var) {
      # for each Variable
      var_df <- .overimp_postprocess(
        obj = obj, vars = var, m = m, imp_idx = imp_idx,
        integerAsFactor = integerAsFactor
      )
      var_vec <- obj$imputed_train[[1]][[var]]
      var_type <- Types[[var]]
      plot_which <- if (var_type == "numeric") num_plot else fac_plot
      plot_fun <- plot_map[[var_type]][[plot_which]]


      if (identical(title, "auto")) {
        title <- default_title
      }
      if (identical(subtitle, "auto")) {
        subtitle <- var
      }

      args_list <- list(
        plot_data = var_df,
        x = var,
        title = title,
        subtitle = subtitle,
        train_color_pal = if (is.null(train_color_pal)) var_df$train$color_pal else train_color_pal,
        test_color_pal = if (is.null(test_color_pal)) var_df$test$color_pal else test_color_pal,
        point_size = params$point_size,
        xlim = params$xlim,
        ylim = params$ylim,
        stack_y = stack_y,
        diag_color = diag_color,
        seed = seed
      )

      do.call(plot_fun, args_list[names(args_list) %in% names(formals(plot_fun))])
    })) |>
    ungroup()


  old_opt <- getOption("progress_enabled")

  on.exit(
    options(progress_enabled = old_opt),
    add = TRUE
  )

  if (isFALSE(verbose)) {
    options(progress_enabled = FALSE)
  }

  if (!is.null(path)) {
    trelliscopejs::trelliscope(all_vars_df, name = "Overimputation diagnostic across all variables", panel_col = "panel", self_contained = FALSE, nrow = nrow, ncol = ncol, path = path)
  } else {
    trelliscopejs::trelliscope(all_vars_df, name = "Overimputation diagnostic across all variables", panel_col = "panel", self_contained = FALSE, nrow = nrow, ncol = ncol)
  }
}
