library(testthat)
library(dplyr)

# helper to create tipse object
make_tipse <- function(dat, method, arm_to_impute, reason, tipping_param) {
  cox_fit <- coxph(Surv(AVAL, EVENT) ~ TRT01P, ties = "exact", data = codebreak200)
  if (method %in% c("hazard inflation", "hazard deflation")) {
    res <- tipping_point_model_based(
      dat = dat,
      reason = reason,
      impute = arm_to_impute,
      J = 2,
      tipping_range = tipping_param,
      cox_fit = cox_fit,
      seed = 123
    )
  } else {
    res <- tipping_point_model_free(
      dat = dat,
      reason = reason,
      impute = arm_to_impute,
      J = 2,
      tipping_range = tipping_param,
      cox_fit = cox_fit,
      method = method,
      seed = 123
    )
  }
}

test_that("assess_plausibility works for control arm with random sampling", {
  tipse <- make_tipse(codebreak200, "random sampling", "docetaxel", "Early dropout", tipping_param = seq(5, 100, by = 10))

  expect_output(
    assess_plausibility(tipse),
    regexp = "median duration of follow-up"
  )
})

test_that("assess_plausibility works for control arm with hazard deflation", {
  tipse <- make_tipse(codebreak200, "hazard deflation", "docetaxel", "Early dropout", tipping_param = seq(0.1, 0.9, by = 0.2))

  expect_output(
    assess_plausibility(tipse),
    regexp = "HR between imputed set in docetaxel arm"
  )
})

test_that("assess_plausibility works for treatment arm with deterministic sampling", {
  tipse <- make_tipse(extenet, "deterministic sampling", "neratinib", "Lost to follow-up", tipping_param = seq(1, 20, by = 1))

  expect_output(
    assess_plausibility(tipse),
    regexp = "event rate in imputed set in neratinib"
  )
})

test_that("assess_plausibility works for treatment arm with hazard inflation", {
  tipse <- make_tipse(extenet, "hazard inflation", "neratinib", "Lost to follow-up", tipping_param = seq(1, 4, by = 0.5))

  expect_output(
    assess_plausibility(tipse),
    regexp = "HR between imputed set in neratinib arm"
  )
})
