% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cf_lm.R
\name{cf_lm}
\alias{cf_lm}
\title{Coarse-to-fine spatial linear modeling}
\usage{
cf_lm(y, x, coords, x0 = NULL, coords0 = NULL, mod_hv)
}
\arguments{
\item{y}{Vector of response variables (N x 1).}

\item{x}{Matrix of covariates (N x K).}

\item{coords}{Matrix of 2-dimensional point coordinates (N x 2).}

\item{x0}{Optional. Matrix of covariates at prediction sites (N0 x K).}

\item{coords0}{Optional. Matrix of 2-dimensional point coordinates at
prediction sites (N0 x 2).}

\item{mod_hv}{Output object of the \code{\link{cf_lm_hv}} function.}
}
\value{
A list with the following elements:
\describe{
  \item{beta}{Regression coefficients, their standard errors, and the lower
  and upper limits of the 95 percent confidence intervals.}
  \item{sd_summary}{Standard deviation of the regression term (xb), spatial
  process (spatial_scale1, spatial_scale2,...),
  additional learning, and residuals.}
  \item{e_summary}{R-squared and RMSE for validation samples, and
  residual standard deviation (residual_SD),
  and root mean squared error for the validation samples (validation_RMSE)}
  \item{pred}{Predictive means and standard deviations (sample sites).}
  \item{pred0}{Predictive means and standard deviations (prediction sites).}
  \item{bands}{Bandwidth values for each scale. The i-th bandwidth is used
  for the spatial process corresponding to the i-th column of the Z matrix).}
  \item{Z}{Predictive mean of the spatial process in each scale
  (sample sites; list).}
  \item{Z_sd}{Predictive standard deviation of the spatial process in each
  scale (sample sites; list).}
  \item{Z0}{Predictive mean of the spatial process in each scale
  (prediction sites; list).}
  \item{Z0_sd}{Predictive standard deviation of the spatial process in each
  scale (prediction sites; list).}
  \item{Other}{Other internal output objects.}
}
}
\description{
Prediction and regression via coarse-to-fine spatial linear modeling.
}
\examples{
set.seed(123)
require(sp); require(sf)
data(meuse)
data(meuse.grid)

### Data
y        <- log(meuse[,"zinc"])
coords   <- meuse[,c("x","y")]
x        <- data.frame(dist   = meuse[,"dist"],
                       ffreq2 = as.integer(meuse$ffreq == 2),
                       ffreq3 = as.integer(meuse$ffreq == 3))

### Data at prediction sites
coords0  <- meuse.grid[,c("x","y")]
x0       <- data.frame(dist   = meuse.grid[,"dist"],
                       ffreq2 = as.integer(meuse.grid$ffreq == 2),
                       ffreq3 = as.integer(meuse.grid$ffreq == 3))

### Holdout validation optimizing the number of spatial scales
mod_hv   <- cf_lm_hv(y = y, x = x, coords = coords, add_learn = "none")

### Spatial modeling and prediction
mod      <- cf_lm(y = y, x = x, x0 = x0, coords = coords, coords0 = coords0,
                 mod_hv = mod_hv)
mod

### Mapping predictive mean and standard deviations (SD)
meuse.grid$pred   <- mod$pred0$pred
meuse.grid$pred_sd<- mod$pred0$pred_sd
meuse.grid_sf     <- st_as_sf(meuse.grid, coords = c("x","y"))
plot(meuse.grid_sf[,"pred"], pch = 15, cex = 0.5, nbreaks = 20)   # Predictive mean
plot(meuse.grid_sf[,"pred_sd"], pch = 15, cex = 0.5, nbreaks = 20)# Predictive SD

### Multiscale spatial pattern/feature extraction
mod_s1<- sp_scalewise(mod,bw_range=c(1000,Inf)) # Large scale (1000 <= bandwdith)
mod_s2<- sp_scalewise(mod,bw_range=c(500,1000)) # Middle scale (500 <= bandwdith <= 1000)
mod_s3<- sp_scalewise(mod,bw_range=c(0,500))    # Small scale (bandwdith <= 500)
z1    <- mod_s1$pred0$pred                      # Predictive mean
z2    <- mod_s2$pred0$pred
z3    <- mod_s3$pred0$pred
z1_sd <- mod_s1$pred0$pred_sd                   # Predictive SD
z2_sd <- mod_s2$pred0$pred_sd
z3_sd <- mod_s3$pred0$pred_sd
meuse.grid_sf3  <- cbind(meuse.grid_sf, z1, z2, z3, z1_sd, z2_sd, z3_sd)
plot(meuse.grid_sf3[,c("z1","z2","z3")], pch = 15,
     cex = 0.5, nbreaks = 20,key.pos=4,axes=TRUE) # Predictive means
plot(meuse.grid_sf3[,c("z1_sd","z2_sd","z3_sd")], pch = 15,
     cex = 0.5, nbreaks = 20,key.pos=4,axes=TRUE) # Predictive SD

}
\references{
Murakami, D., Comber, A., Yoshida, T., Tsutsumida, N., Brunsdon, C.,
& Nakaya, T. (2025).
Coarse-to-fine spatial modeling: A scalable, machine-learning-compatible
spatial model.
*arXiv:2510.00968*.
}
\seealso{
\code{\link{cf_lm_hv}}, \code{\link{sp_scalewise}}
}
\author{
Daisuke Murakami
}
