#' Decode reference and read into aligned sequences according to CIGAR string
#'
#' @param cigar A CIGAR string
#' @param ref The reference sequence
#' @param reads The read sequence
#' 
#' @return A data frame of alignment
#' 
#' @examples
#' \dontrun{
#'     decodeCIGAR()
#' }
#'
#' @noRd
decodeCIGAR <- function(cigar, ref, reads){
    
	op <- parseCIGAR(cigar)
	aln.df <- data.frame( 
		cigar = rep(op$code, op$length),
		cigar_index = 1:sum(op$length),
		stringsAsFactors = FALSE 
	)
	
	ref.df <- aln.df[ aln.df$cigar %in% c("M","D","N","=","X"), ]
	ref.df$ref_index <- 1:nrow(ref.df)
	ref.df$ref_seq <- unlist( strsplit(ref, "") )
	ref.df$cigar <- NULL
	
	reads.df <- aln.df[ aln.df$cigar %in% c("M","I","=","X"), ]
	reads.df$reads_index <- 1:nrow(reads.df)
	reads.df$reads_seq <- unlist( strsplit(reads, "") )
	reads.df$cigar <- NULL
	
	aln.df <- merge(aln.df, ref.df, by="cigar_index", all.x=TRUE)
	aln.df <- merge(aln.df, reads.df, by="cigar_index", all.x=TRUE)	
	aln.df$ref_seq[ aln.df$cigar=="I" ] <- "-"
	aln.df$reads_seq[ aln.df$cigar=="D" ] <- "-"
	aln.df$reads_seq[ aln.df$cigar=="N" ] <- "*"
	aln.df <- aln.df[ order(aln.df$cigar_index), ]
	
	cigar.str <- paste(aln.df$cigar, collapse = "")
	aligned.reads <- paste(aln.df$reads_seq, collapse = "")
	aligned.ref <- paste(aln.df$ref_seq, collapse = "")
	
	aln.df$snv <- ifelse( 
		aln.df$cigar%in%c("M","=","X") & aln.df$ref_seq!=aln.df$reads_seq, 
		paste(aln.df$ref_seq,aln.df$reads_seq, sep=">"),
		NA 
	)
	
	aln.df$del <- NA
	aln.df$ins <- NA
	
	match.D <- gregexpr("D+", cigar.str)
	if(match.D[[1]][1]!=-1){
		df.D <- data.frame(
			code = "D",
			length = attr(match.D[[1]],"match.length"),
			pos = match.D[[1]],
			stringsAsFactors = FALSE
		)
		aln.df$del[df.D$pos] <- df.D$length
	}
	
	match.I <- gregexpr("MI+", cigar.str)
	if(match.I[[1]][1]!=-1){
		df.I <- data.frame(
			code = "I",
			length = attr(match.I[[1]],"match.length")-1,
			pos = match.I[[1]],
			stringsAsFactors = FALSE
		)
		match.I[[1]] <- match.I[[1]] + 1
		attr(match.I[[1]],"match.length") <- attr(match.I[[1]],"match.length") - 1
		df.I$seq <- unlist( regmatches(aligned.reads, match.I) )
		aln.df$ins[df.I$pos] <- df.I$seq
	}
	
	match.DI <- gregexpr("D+I+", cigar.str)
	if(match.DI[[1]][1]!=-1){
	    delins.df <- data.frame(
	        delins_pos = match.DI[[1]],
	        delins_len = attr(match.DI[[1]],"match.length")
	    )
	    del.df <- data.frame(
	        delins_pos = match.D[[1]],
	        del_len = attr(match.D[[1]],"match.length")
	    )
	    delins.df <- merge( delins.df, del.df, by="delins_pos" )
	    delins.df$ins_pos <- delins.df$delins_pos + delins.df$del_len
	    delins.df$ins_len <- delins.df$delins_len - delins.df$del_len
	    
	    match.DI.ins <- match.DI
	    match.DI.ins[[1]] <- delins.df$ins_pos
	    attr(match.DI.ins[[1]],"match.length") <- delins.df$ins_len
	    
	    delins.df$ins_seq <- unlist( regmatches(aligned.reads, match.DI.ins) )
	    aln.df$ins[delins.df$delins_pos] <- delins.df$ins_seq
	}
	
	return(aln.df)
}
