% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/describe.R
\name{describe_imputation}
\alias{describe_imputation}
\title{Describe an rbmi Imputation Object}
\usage{
describe_imputation(impute_obj)
}
\arguments{
\item{impute_obj}{An \code{imputation} object returned by \code{\link[rbmi:impute]{rbmi::impute()}}.}
}
\value{
An S3 object of class \code{c("describe_imputation", "list")} containing:
\describe{
\item{method}{Human-readable method name (e.g., "Bayesian (MCMC via Stan)")}
\item{method_class}{Raw class name: "bayes", "approxbayes", or "condmean"}
\item{n_imputations}{Number of imputations (M)}
\item{references}{Named character vector of reference arm mappings, or NULL}
\item{n_subjects}{Total number of unique subjects}
\item{visits}{Character vector of visit names}
\item{missingness}{A data.frame with columns: visit, group, n_total,
n_miss, pct_miss}
}
}
\description{
Extracts structured metadata from an rbmi imputation object, including
method, number of imputations (M), reference arm mappings, subject count,
and a missingness breakdown by visit and treatment arm. Returns an S3 object
with an informative \code{\link[=print]{print()}} method.
}
\details{
The missingness table is built by cross-tabulating
\code{impute_obj$data$is_missing} by visit and treatment group. Each row shows
the total number of subjects in that group, how many had missing data at
that visit, and the percentage missing.
}
\examples{
\dontrun{
library(rbmi)
library(dplyr)
data("ADEFF", package = "rbmiUtils")

ADEFF <- ADEFF |>
  mutate(
    TRT = factor(TRT01P, levels = c("Placebo", "Drug A")),
    USUBJID = factor(USUBJID),
    AVISIT = factor(AVISIT, levels = c("Week 24", "Week 48"))
  )

vars <- set_vars(
  subjid = "USUBJID", visit = "AVISIT", group = "TRT",
  outcome = "CHG", covariates = c("BASE", "STRATA", "REGION")
)
dat <- ADEFF |> select(USUBJID, STRATA, REGION, TRT, BASE, CHG, AVISIT)
draws_obj <- draws(
  data = dat, vars = vars,
  method = method_bayes(n_samples = 100)
)
impute_obj <- impute(
  draws_obj,
  references = c("Placebo" = "Placebo", "Drug A" = "Placebo")
)

# Inspect the imputation
desc <- describe_imputation(impute_obj)
print(desc)

# Programmatic access
desc$n_imputations
desc$missingness
desc$references
}

}
\seealso{
\itemize{
\item \code{\link[rbmi:impute]{rbmi::impute()}} to create imputation objects
\item \code{\link[=describe_draws]{describe_draws()}} for inspecting draws objects
}
}
