% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spec_rodrigues_targets.R
\name{spec_rodrigues_targets}
\alias{spec_rodrigues_targets}
\title{Specify targets following Rodrigues \emph{et al.} (2004)}
\usage{
spec_rodrigues_targets(
  rare_area_threshold = 1000,
  rare_relative_target = 1,
  common_area_threshold = 250000,
  common_relative_target = 0.1,
  cap_area_target = 1e+06,
  area_units = "km^2"
)
}
\arguments{
\item{rare_area_threshold}{\code{numeric} value indicating the threshold area
for rare identifying rare features.
Defaults to 1000 (i.e., 1000 \ifelse{html}{\out{km<sup>2</sup>}}{\eqn{km^2}}).}

\item{rare_relative_target}{\code{numeric} value denoting the
relative target for features with a spatial distribution
that is smaller than \code{rare_area_threshold}.
Note that this value must be a proportion between 0 and 1.
Defaults to 1 (i.e., 100\%).}

\item{common_area_threshold}{\code{numeric} value indicating the threshold area
for identifying common features.
Defaults to 250000
(i.e., 250,000 \ifelse{html}{\out{km<sup>2</sup>}}{\eqn{km^2}}).}

\item{common_relative_target}{\code{numeric} value denoting the
relative target for features with a spatial distribution
that is greater than \code{common_area_threshold}.
Defaults to 0.1 (i.e., 10\%).
Since this default value is based on historical levels of global protected
area coverage, it may be appropriate to set this value based on current
levels of protected area coverage (e.g., 17.6\% for terrestrial and 8.4\% for
marine systems globally; UNEP-WCMC and IUCN 2025).}

\item{cap_area_target}{\code{numeric} value denoting the area-based target cap.
To avoid setting a target cap, a missing (\code{NA}) value can be specified.
Defaults to 1000000
(i.e., 1,000,000 \ifelse{html}{\out{km<sup>2</sup>}}{\eqn{km^2}}).}

\item{area_units}{\code{character} value denoting the unit of measurement
for the area-based arguments.
Defaults to \code{"km^2"}
(i.e., \ifelse{html}{\out{km<sup>2</sup>}}{\eqn{km^2}}).}
}
\value{
An object (\code{\linkS4class{TargetMethod}}) for specifying targets that
can be used with \code{\link[=add_auto_targets]{add_auto_targets()}} and \code{\link[=add_group_targets]{add_group_targets()}}
to add targets to a \code{\link[=problem]{problem()}}.
}
\description{
Specify targets based on the methodology outlined by
Rodrigues \emph{et al.} (2004).
Briefly, this method involves setting targets based on log-linear
interpolation methods.
To help prevent widespread features from obscuring priorities,
targets are capped following Butchart \emph{et al.} (2015).
This method was designed for global-scale prioritizations.
Note that this function is designed to be used with \code{\link[=add_auto_targets]{add_auto_targets()}}
and \code{\link[=add_group_targets]{add_group_targets()}}.
}
\details{
This target setting method was designed to protect species in global-scale
prioritizations (Rodrigues \emph{et al.} 2004).
Although it has also been successfully applied to global-scales
(e.g., Butchart \emph{et al.} 2015; Hanson \emph{et al.} 2020; Venter \emph{et al.} 2014),
it may fail to identify meaningful priorities for
prioritizations conducted at smaller geographic scales
(e.g., national, state-level or county scales).
For example, if this method is applied to
such geographic scales, then the resulting prioritizations
may select an overly large percentage of the study area,
or be biased towards over-representing common and widespread species.
This is because the thresholds
(i.e., \code{rare_area_threshold}, \code{common_area_threshold},
and \code{cap_area_threshold})
were originally developed based on rationale for promoting the long-term
persistence of entire species.
As such, if you are working at a sub-global scale, it is recommended to set
thresholds based on that criteria are appropriate to the spatial extent
of the planning region.
Please note that this function is provided as convenient method to
set targets for problems with a single management zone, and cannot
be used for those with multiple management zones.
}
\section{Mathematical formulation}{

This method involves setting target thresholds based on the spatial
extent of the features.
By default, this method identifies rare features as those with a
spatial distribution smaller than 1,000
\ifelse{html}{\out{km<sup>2</sup>}}{\eqn{km^2}}
(per \code{rare_area_threshold} and \code{area_units})
and common features as those with a spatial distribution
larger than 250,000
\ifelse{html}{\out{km<sup>2</sup>}}{\eqn{km^2}}
(per \code{common_area_threshold} and \code{area_units}).
Given this, rare features are assigned a target threshold
of 100\% (per \code{rare_relative_target}), common features
are assigned a target threshold of 10\% (per \code{common_relative_target}),
and features with a spatial distribution that is between
the area-based thresholds used to identify rare and common features are
assigned a target threshold through log-linear interpolation.
Additionally, following Butchart \emph{et al.} (2015), a cap of 1,000,000
\ifelse{html}{\out{km<sup>2</sup>}}{\eqn{km^2}} is applied to target
thresholds (per \code{cap_area_threshold} and \code{area_units}).
}

\section{Data calculations}{

This function involves calculating targets based on the spatial extent
of the features in \code{x}.
Although it can be readily applied to \code{\link[=problem]{problem()}} objects that
have the feature data provided as a \code{\link[terra:rast]{terra::rast()}} object,
you will need to specify the spatial units for the features
when initializing the \code{\link[=problem]{problem()}} objects if the feature data
are provided in a different format. In particular, if the feature
data are provided as a \code{data.frame} or \code{character} vector,
then you will need to specify an argument to \code{feature_units} when
using the \code{\link[=problem]{problem()}} function.
See the Examples section of the documentation for \code{\link[=add_auto_targets]{add_auto_targets()}}
for a demonstration of specifying the spatial units for features.
}

\examples{
\dontrun{
# set seed for reproducibility
set.seed(500)

# load data
sim_complex_pu_raster <- get_sim_complex_pu_raster()
sim_complex_features <- get_sim_complex_features()

# create problem with Rodrigues et al. (2004) targets
p1 <-
  problem(sim_complex_pu_raster, sim_complex_features) \%>\%
  add_min_set_objective() \%>\%
  add_auto_targets(method = spec_rodrigues_targets()) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# solve problem
s1 <- solve(p1)

# plot solution
plot(s1, main = "solution", axes = FALSE)
}
}
\references{
Butchart SHM, Clarke M, Smith RJ, Sykes RE, Scharlemann JPW, Harfoot M,
Buchanan GM, Angulo A, Balmford A, Bertzky B, Brooks TM, Carpenter KE,
Comeros‐Raynal MT, Cornell J, Ficetola GF, Fishpool LDC, Fuller RA,
Geldmann J, Harwell H, Hilton‐Taylor C, Hoffmann M, Joolia A, Joppa L,
Kingston N, May I, Milam A, Polidoro B, Ralph G, Richman N, Rondinini C,
Segan DB, Skolnik B, Spalding MD, Stuart SN, Symes A, Taylor J, Visconti P,
Watson JEM, Wood L, Burgess ND (2015) Shortfalls and solutions for meeting
national and global conservation area targets. \emph{Conservation Letters},
8: 329--337.

Hanson JO, Rhodes JR, Butchart SHM, Buchanan GM, Rondinini C, Ficetola GF,
Fuller RA (2020) Global conservation of species' niches. Global conservation
of species' niches. \emph{Nature}, 580: 232--234

Rodrigues ASL, Akçakaya HR, Andelman SJ, Bakarr MI, Boitani L, Brooks TM,
Chanson JS, Fishpool LDC, Da Fonseca GAB, Gaston KJ, Hoffmann M, Marquet PA,
Pilgrim JD, Pressey RL, Schipper J, Sechrest W, Stuart SN, Underhill LG,
Waller RW, Watts MEJ, Yan X (2004)
Global gap analysis: priority regions for expanding the global
protected-area network. \emph{BioScience}, 54: 1092--1100.

UNEP-WCMC and IUCN (2025) Protected Planet Report 2024.
Cambridge, UK: UNEP-WCMC and IUCN. Available at <www.protectedplanet.net>.

Venter O, Fuller RA, Segan DB, Carwardine J, Brooks T, Butchart SHM,
Di Marco M, Iwamura T, Joseph L, O'Grady D, Possingham HP, Rondinini C,
Smith RJ, Venter M, Watson JEM (2014) Targeting global protected area
expansion for imperiled biodiversity. \emph{PLoS Biology}, 12: e1001891.
}
\seealso{
Other target setting methods:
\code{\link{spec_absolute_targets}()},
\code{\link{spec_area_targets}()},
\code{\link{spec_duran_targets}()},
\code{\link{spec_interp_absolute_targets}()},
\code{\link{spec_interp_area_targets}()},
\code{\link{spec_jung_targets}()},
\code{\link{spec_max_targets}()},
\code{\link{spec_min_targets}()},
\code{\link{spec_polak_targets}()},
\code{\link{spec_pop_size_targets}()},
\code{\link{spec_relative_targets}()},
\code{\link{spec_rl_ecosystem_targets}()},
\code{\link{spec_rl_species_targets}()},
\code{\link{spec_rule_targets}()},
\code{\link{spec_ward_targets}()},
\code{\link{spec_watson_targets}()},
\code{\link{spec_wilson_targets}()}
}
\concept{methods}
