% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parameterTransform.R
\name{parameterTransform}
\alias{parameterTransform}
\title{Automated transformations of model nodes to unconstrained scales}
\usage{
parameterTransform(model, nodes = character(0), control = list())
}
\arguments{
\item{model}{A nimble model object.  See details.}

\item{nodes}{A character vector specifying model node names to undergo transformation.  See details.}

\item{control}{An optional list allowing for additional control of the transformation. This currently supports a single element \code{allowDeterm}.}
}
\description{
Provide general transformations of constrained continuous-valued model nodes (parameters) to unconstrained scales.  It handles the cases of interval-bounded parameters (e.g. uniform or beta distributions), semi-interval-bounded parameters (e.g. exponential or gamma distributions), and the multivariate Wishart, inverse Wishart, Dirichlet, and LKJ distributions.  Utilities are provided to transform parameters to an unconstrained scale, back-transform from the unconstrained scale to the original scale of the constrained parameterization, and to calculate the natural logarithm of the determinant of the Jacobian matrix of the inverse transformation, calculated at any location in the transformed (unconstrained) space.
}
\details{
The \code{parameterTransform} nimbleFunction is an unspecialized function.  Calling \code{parameterTransform(model, nodes)} will generate and return a specialized nimbleFunction, which provides transformation functionality for the specified hierarchical model and set of model nodes.  The \code{nodes} argument can represent mutliple model nodes arising from distinct prior distributions, which will be simultaneously transformed according to their respective distributions and constraints.

If the \code{nodes} argument is missing or has length zero, then no nodes will be transformed.  A specialized nimbleFunction is created, but will not transform or operate on any model nodes.

The \code{control} argument is a list that supports one additional setting. If \code{control$allowDeterm=FALSE} (the default), deterministic nodes are not allowed in the \code{nodes} argument.  If \code{control$allowDeterm=TRUE}, deterministic nodes are allowed and assumed to have no constraints on valid values.

This specialized nimbleFunction has the following methods:

\code{transform}: Transforms a numeric vector of values from the original constrained model scale to a vector of values on the unconstrained scale.

\code{inverseTransform}: Transforms a numeric vector of values from the unconstrained scale to the original constrained parameterization scale.

The unconstrained scale may have different dimensionality from the original constrained scale of the model parameters.  For example, a d-dimensional dirichlet distribution is constrained to reside on a simplex in d-dimensional space.  In contrast, the corresponding unconstrained parameterization is unrestrained in (d-1) dimensional space.  The specialized \code{parameterTransform} nimbleFunction also provides utilities to return the dimensionality of the original (constrained) parameterization, and the transformed (unconstrained) parameterization:

\code{getOriginalLength}: Returns the dimensionality (number of scalar elements) of the original constrained parameterization.

\code{getTransformedLength}: Returns the dimensionality (number of scalar elements) comprising the transformed unconstrained parameterization.

The specialized \code{parameterTransform} nimbleFunction also provides a method for calculating the natural logarithm of the jacobian of the inverse transformation, calculated at any point in the transformed (unconstrained) space:

\code{logDetJacobian}

The \code{parameterTransformation} function has no facility for handling discrete-valued parameters.
}
\examples{
\dontrun{
code <- nimbleCode({
    a ~ dnorm(0, 1)
    b ~ dgamma(1, 1)
    c ~ dunif(2, 10)
    d[1:3] ~ dmnorm(mu[1:3], cov = C[1:3,1:3])
    e[1:3,1:3] ~ dwish(R = C[1:3,1:3], df = 5)
})
 
constants <- list(mu=rep(0,3), C=diag(3))
 
Rmodel <- nimbleModel(code, constants)
 
## create a specialized parameterTransform function:
nodes <- c('a', 'b', 'c', 'd', 'e')
pt <- parameterTransform(Rmodel, nodes)
 
vals <- c(1, 10, 5,    1,2,3,   as.numeric(diag(3)))
 
## transform values to unconstrained scale:
transformedVals <- pt$transform(vals)
 
## back-transform to original constrained scale of parameterization
pt$inverseTransform(transformedVals)  ## return is same as original vals
 
## dimensionality of original constrained scale = 1 + 1 + 1 + 3 + 9
pt$getOriginalLength()      ## 15
 
## dimensionality of transformed (unconstrained) scale = 1 + 1 + 1 + 3 + 6
pt$getTransformedLength()   ## 12
 
## log of the jacobian of the inverse transformation matrix:
pt$logDetJacobian(transformedVals)
}

}
\seealso{
\code{\link{buildLaplace}}
}
\author{
Daniel Turek
}
