#' Create request for Meteoclimatic API
#'
#' Create the request for meteoclimatic based on path
#'
#' This function creates the request, deal with errors and retries if needed
#' and access the data of the response.
#'
#' @param path character vector with the path as obtained by
#'   \link{\code{.create_meteoclimatic_path}}
#' @param api_options Option list as generated by \link{\code{metoclimatic_options}}
#'
#' @return the response xml.
#'
#' @noRd
.create_meteoclimatic_request <- function(path) {
  meteoclimatic_request <- httr2::request("https://meteoclimatic.net/feed") |>
    httr2::req_url_path_append(path) |>
    httr2::req_user_agent(
      "meteospain R package (https://emf.creaf.cat/software/meteospain/)"
    ) |>
    httr2::req_error(
      body = \(resp) {
        message <- httr2::resp_status(resp)
        
        # more verbose known errors
        # if (httr2::resp_status(resp) %in% c(404L)) {
        #   message <- c(
        #     message,
        #     "This usually happens when unknown station ids are supplied."
        #   )
        # }

        return(message)
      }
    ) |>
    httr2::req_retry(
      max_tries = 3,
      retry_on_failure = TRUE,
      is_transient = \(resp) {
        httr2::resp_status(resp) %in% c(429, 503)
      },
      backoff = \(resp) {
        60
      },
      after = \(resp) {
        NA
      }
    )
  httr2::req_perform(meteoclimatic_request) |>
    httr2::resp_body_xml()
}

#' Create the path elements for Meteoclimatic API
#'
#' Path vectors for Meteoclimatic API to use with httr2
#'
#' @param api_options Option list as generated by \link{\code{meteoclimatic_options}}
#'
#' @noRd
.create_meteoclimatic_path <- function(api_options) {

  meteoclimatic_path <- c("xml", api_options$stations)

  # not recognised resolution
  if (api_options$resolution != 'current_day') {
    cli::cli_abort(c(
      "{.arg {api_options$resolution}} is not a valid temporal resolution for Meteoclimatic. Please see meteoclimatic_options help for more information"
    ))
  }

  meteoclimatic_path
}

#' Get info for the meteoclimatic stations
#'
#' Get info for the meteoclimatic stations
#'
#' @noRd

.get_info_meteoclimatic <- function(api_options) {
  # path
  path_resolution <- c("rss", api_options$stations)
  # station_info
  raw_station_info <- .create_meteoclimatic_request(path_resolution)
  dplyr::tibble(
    service = 'meteoclimatic',
    # station_id is special, we need obtain it from the link, so we need to remove all the link previous to the code
    station_id = substr(xml2::xml_text(xml2::xml_find_all(raw_station_info, '//item/link')), start = 37, stop = 100),
    # name, lat and long are elements in each node/item, so we extract them
    station_name = xml2::xml_text(xml2::xml_find_all(raw_station_info, '//item/title')),
    lat = xml2::xml_double(xml2::xml_find_all(raw_station_info, '//item/geo:Point/geo:lat')),
    long = xml2::xml_double(xml2::xml_find_all(raw_station_info, '//item/geo:Point/geo:long'))
  ) |>
    sf::st_as_sf(coords = c('long', 'lat'), crs = 4326)
}

#' Get data from Meteoclimatic
#'
#' Get data from Meteoclimatic service
#'
#' Only current day (hourly data) will be returned
#'
#' @param api_options Option list as generated by \link{\code{meteclimatic_options}}
#'
#' @noRd
.get_data_meteoclimatic <- function(api_options) {

  # api path
  path_resolution <- .create_meteoclimatic_path(api_options)

  # cache
  # NOTE: meteoclimatic only offers current day resolution, so no cache is
  # used.

  # data
  data_xml_body <- .create_meteoclimatic_request(path_resolution)
  # data is an xml file. Each station is a node in the xml, so we loop by nodes and build the tibble. Finally
  # we need the stations info, so we join it
  nodes <- xml2::xml_path(xml2::xml_find_all(data_xml_body, '//meteodata/stations/station'))
  # but before start iterating, if station code is wrong, no nodes are returned, so we need to check that
  if (length(nodes) < 1) {
    cli::cli_abort(c(
      'Stations code provided',
      api_options$stations,
      'not found in Meteoclimatic database.',
      i = "See https://www.meteoclimatic.net/index/wp/rss_es.html for more info"
    ))
  }

  # Now we can iterate and get the data
  stations_data <-
    nodes |>
    purrr::map(
      \(.x) {
        dplyr::tibble(
          service = 'meteoclimatic',
          timestamp = lubridate::parse_date_time(
            xml2::xml_text(xml2::xml_find_first(data_xml_body, paste0(.x, '/pubDate'))),
            orders = 'dbYHMSz'
          ),
          station_id = xml2::xml_text(
            xml2::xml_find_first(data_xml_body, paste0(.x, '/id'))
          ),
          max_temperature = xml2::xml_double(
            (xml2::xml_find_first(data_xml_body, paste0(.x, '/stationdata/temperature/max')))
          ),
          min_temperature = xml2::xml_double(
            (xml2::xml_find_first(data_xml_body, paste0(.x, '/stationdata/temperature/min')))
          ),
          max_relative_humidity = xml2::xml_double(
            (xml2::xml_find_first(data_xml_body, paste0(.x, '/stationdata/humidity/max')))
          ),
          min_relative_humidity = xml2::xml_double(
            (xml2::xml_find_first(data_xml_body, paste0(.x, '/stationdata/humidity/min')))
          ),
          precipitation = xml2::xml_double(
            (xml2::xml_find_first(data_xml_body, paste0(.x, '/stationdata/rain/total')))
          ),
          max_atmospheric_pressure = xml2::xml_double(
            (xml2::xml_find_first(data_xml_body, paste0(.x, '/stationdata/barometre/max')))
          ),
          min_atmospheric_pressure = xml2::xml_double(
            (xml2::xml_find_first(data_xml_body, paste0(.x, '/stationdata/barometre/min')))
          ),
          wind_direction = xml2::xml_double(
            (xml2::xml_find_first(data_xml_body, paste0(.x, '/stationdata/wind/azimuth')))
          ),
          max_wind_speed = xml2::xml_double(
            (xml2::xml_find_first(data_xml_body, paste0(.x, '/stationdata/wind/max')))
          )
        )
      }
    ) |>
    purrr::list_rbind() |>
    dplyr::left_join(.get_info_meteoclimatic(api_options), by = c('service', 'station_id')) |>
    dplyr::select("timestamp", "station_id", "station_name", dplyr::everything()) |>
    dplyr::mutate(
      max_temperature = units::set_units(.data$max_temperature, "degree_C"),
      min_temperature = units::set_units(.data$min_temperature, "degree_C"),
      max_relative_humidity = units::set_units(.data$max_relative_humidity, "%"),
      min_relative_humidity = units::set_units(.data$min_relative_humidity, "%"),
      precipitation = units::set_units(.data$precipitation, "L/m^2"),
      max_atmospheric_pressure = units::set_units(.data$max_atmospheric_pressure, "hPa"),
      min_atmospheric_pressure = units::set_units(.data$min_atmospheric_pressure, "hPa"),
      wind_direction = units::set_units(.data$wind_direction, "degree"),
      max_wind_speed = units::set_units(.data$max_wind_speed, "km/h")
    ) |>
    dplyr::arrange(.data$timestamp, .data$station_id) |>
    # reorder variables to be consistent among all services
    relocate_vars() |>
    sf::st_as_sf(crs = 4326)

  # Copyright message -------------------------------------------------------------------------------------
  cli::cli_inform(c(
    i = copyright_style("Meteoclimatic is a non-professional network of automatic meteorological stations."),
    copyright_style("No quality check is performed in this data, and errors in measures or coordinates of stations can be present."),
    legal_note_style("https://www.meteoclimatic.net/index")
  ))

  stations_data
}
