% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sf_fisheye.R
\name{sf_fisheye}
\alias{sf_fisheye}
\title{Radial fisheye warp for \code{sf}/\code{sfc} objects (auto-CRS + flexible centers)}
\usage{
sf_fisheye(
  sf_obj,
  center = NULL,
  center_crs = NULL,
  normalized_center = FALSE,
  cx = NULL,
  cy = NULL,
  r_in = 0.34,
  r_out = 0.5,
  zoom_factor = 1.5,
  squeeze_factor = 0.35,
  method = "expand",
  revolution = 0,
  target_crs = NULL,
  preserve_aspect = TRUE
)
}
\arguments{
\item{sf_obj}{An \code{\link[sf:sf]{sf}} or \code{\link[sf:sfc]{sfc}} object. Supports
\code{POINT}, \code{LINESTRING}, \code{POLYGON}, and \code{MULTIPOLYGON}. Empty geometries
are removed before processing.}

\item{center}{Flexible center specification (see \strong{Center selection}):
\itemize{
\item numeric length-2 pair interpreted via \code{center_crs} or by lon/lat
heuristic, or as map units if not lon/lat;
\item any \code{sf}/\code{sfc} geometry, from which a centroid is derived;
\item normalized \eqn{[-1,1]} pair when \code{normalized_center = TRUE}.
}}

\item{center_crs}{Optional CRS for a numeric \code{center} (e.g., \code{"EPSG:4326"}).
Ignored if \code{center} is an \code{sf}/\code{sfc} object (its own CRS is used).}

\item{normalized_center}{Logical. If \code{TRUE}, \code{center} is treated as a
normalized \eqn{[-1,1]} coordinate around the bbox midpoint.}

\item{cx, cy}{Optional center in \strong{working CRS} map units (legacy path,
ignored when \code{center} is provided).}

\item{r_in, r_out}{Numeric radii (in \strong{normalized units}) defining focus and
glue boundaries; must satisfy \code{r_out > r_in}.}

\item{zoom_factor}{Numeric (> 1 to enlarge). Focus magnification passed to
\code{fisheye_fgc()}.}

\item{squeeze_factor}{Numeric in [0, 1]. Glue-zone compression strength
passed to \code{fisheye_fgc()}.}

\item{method}{Character; name understood by \code{fisheye_fgc()} (default \code{"expand"}).}

\item{revolution}{Numeric (radians); optional angular twist for glue zone,
passed to \code{fisheye_fgc()}.}

\item{target_crs}{Optional working CRS (anything accepted by
\code{\link[sf:st_crs]{sf::st_crs()}} / \code{\link[sf:st_transform]{sf::st_transform()}}). If \code{NULL}, a projected CRS is
auto-selected when the input is lon/lat; otherwise the input CRS is used.}

\item{preserve_aspect}{Logical. If \code{TRUE} (default), use uniform scaling; if
\code{FALSE}, scale axes independently (may stretch shapes).}
}
\value{
An object of the same top-level class as \code{sf_obj} (\code{sf} or \code{sfc}),
with geometry coordinates warped by the fisheye and the \strong{original CRS}
restored.
}
\description{
\code{sf_fisheye()} applies a \strong{focus–glue–context} fisheye to vector data:
it (1) ensures a sensible projected working CRS, (2) \strong{normalizes}
coordinates around a chosen center, (3) calls \code{fisheye_fgc()} to warp radii,
(4) \strong{denormalizes} back to map units, and (5) restores the original CRS.
Inside the focus ring (\code{r_in}) features enlarge; across the glue ring
(\code{r_out}) they transition smoothly; outside, they stay nearly unchanged.
}
\details{
\strong{CRS handling.} If \code{target_crs} is \code{NULL} and the input is geographic
(lon/lat), a projected \strong{working CRS} is chosen from the layer’s centroid:
\itemize{
\item Victoria, AU region (approximate 140–150°E, 40–30°S): \strong{EPSG:7855} (GDA2020 / MGA55).
\item Otherwise UTM: \strong{EPSG:326##} (north) or \strong{EPSG:327##} (south).
}
You may override with \code{target_crs}. The original CRS is restored on return.

\strong{Center selection.} The fisheye center can be supplied in multiple ways:
\itemize{
\item \code{center = c(lon, lat)}, with \code{center_crs = "EPSG:4326"} (recommended
for WGS84) or another CRS string/object.
\item \code{center = c(x, y)} already in \strong{working CRS} map units (meters).
\item \code{center} as any \code{sf}/\code{sfc} geometry (POINT/LINE/POLYGON/etc.): its
\strong{centroid of the combined geometry} is used, then transformed to the
working CRS.
\item \code{center = c(cx, cy)} as \strong{normalized} coordinates in \eqn{[-1,1]}
when \code{normalized_center = TRUE} (relative to the bbox midpoint and
scale used for normalization).
\item Legacy \verb{cx, cy} (map units) are still accepted and used only when
\code{center} is not supplied.
}

\strong{Normalization.} Let bbox half-width/height be \code{sx}, \code{sy}. With
\code{preserve_aspect = TRUE} (default), a uniform scale \code{s = max(sx, sy)} maps
\eqn{(x, y) \mapsto ((x - cx)/s,\, (y - cy)/s)}, so \code{r_in}/\code{r_out} (e.g.,
0.34/0.5) are interpreted in a unit-like space. If \code{preserve_aspect = FALSE},
X and Y are independently scaled by \code{sx} and \code{sy}.

\strong{Implementation notes.} Geometry coordinates are transformed by
\code{st_transform_custom()} which safely re-closes polygon rings and drops Z/M.
The radial warp itself is delegated to \code{fisheye_fgc()} (which is not modified).

The transformation may introduce self-intersections or other topology
issues due to geometric warping. To ensure the output is suitable for
plotting and spatial operations, the geometry is repaired using
\code{lwgeom::lwgeom_make_valid()}. Users should be aware that:
\itemize{
\item geometry types may be promoted (e.g., POLYGON → MULTIPOLYGON),
\item tiny sliver polygons may be removed,
\item invalid rings or bow-tie shapes will be corrected,
\item the repair step requires the \code{{lwgeom}} package.
}
}
\examples{
library(sf)

# Toy polygon in a projected CRS
poly <- st_sfc(st_polygon(list(rbind(
  c(0,0), c(1,0), c(1,1), c(0,1), c(0,0)
))), crs = 3857)

# Default center (bbox midpoint), gentle magnification
out1 <- sf_fisheye(poly, r_in = 0.3, r_out = 0.6,
                   zoom_factor = 1.5, squeeze_factor = 0.35)

# Explicit map-unit center, stronger focus
out2 <- sf_fisheye(poly, cx = 0.5, cy = 0.5,
                   r_in = 0.25, r_out = 0.55,
                   zoom_factor = 2.0, squeeze_factor = 0.25)

# Lon/lat point (auto-project to UTM/MGA), then fisheye around CBD (WGS84)
pt_ll <- st_sfc(st_point(c(144.9631, -37.8136)), crs = 4326)  # Melbourne CBD
out3  <- sf_fisheye(pt_ll, r_in = 0.2, r_out = 0.5)

# Center supplied as an sf polygon: centroid is used as the warp center
out4 <- sf_fisheye(poly, center = poly)

}
\seealso{
\code{\link[sf:st_transform]{sf::st_transform()}}, \code{\link[sf:st_is_longlat]{sf::st_is_longlat()}}, \code{\link[sf:st_crs]{sf::st_crs()}},
\code{\link[sf:st_coordinates]{sf::st_coordinates()}}, \code{st_transform_custom()}, \code{fisheye_fgc()}
\code{\link[lwgeom:lwgeom_make_valid]{lwgeom::lwgeom_make_valid()}}, \code{\link[sf:valid]{sf::st_make_valid()}}
}
