% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_process_parameters.R
\name{estimate_process_parameters}
\alias{estimate_process_parameters}
\title{Estimate point process parameters using log-likelihood maximization}
\usage{
estimate_process_parameters(
  data,
  process = c("self_correcting"),
  grids,
  budgets,
  parameter_inits = NULL,
  delta = NULL,
  parallel = FALSE,
  num_cores = max(1L, parallel::detectCores() - 1L),
  set_future_plan = FALSE,
  strategy = c("local", "global_local", "multires_global_local"),
  global_algorithm = "NLOPT_GN_CRS2_LM",
  local_algorithm = "NLOPT_LN_BOBYQA",
  starts = list(global = 1L, local = 1L, jitter_sd = 0.35, seed = 1L),
  finite_bounds = NULL,
  refine_best_delta = TRUE,
  rescore_control = list(enabled = TRUE, top = 5L, objective_tol = 1e-06, param_tol =
    0.1, avoid_bound_solutions = TRUE, bound_eps = 1e-08),
  verbose = TRUE
)
}
\arguments{
\item{data}{a data.frame or matrix. Must contain either columns \code{(time, x, y)}
or \code{(x, y, size)}. If a matrix is provided without time, it must have
column names \code{c("x","y","size")}.}

\item{process}{type of process used (currently supports \code{"self_correcting"}).}

\item{grids}{a \code{\link{ldmppr_grids}} object specifying the integration grid schedule
(single-level or multi-resolution). The integration bounds are taken from
\code{grids$upper_bounds}.}

\item{budgets}{a \code{\link{ldmppr_budgets}} object controlling optimizer options
for the global stage and local stages (first level vs refinement levels).}

\item{parameter_inits}{(optional) numeric vector of length 8 giving initialization values
for the model parameters. If \code{NULL}, defaults are derived from \code{data} and
\code{grids$upper_bounds}.}

\item{delta}{(optional) numeric scalar or vector. Used only when \code{data} does not contain
\code{time} (i.e., data has \code{(x,y,size)}).
\itemize{
  \item If \code{length(delta) == 1}, fits the model once using \code{power_law_mapping(size, delta)}.
  \item If \code{length(delta) > 1}, performs a delta-search by fitting the model for each candidate value
    and selecting the best objective. If \code{refine_best_delta = TRUE} and multiple grid levels are used,
    the best delta is refined on the remaining (finer) grid levels.
}
If \code{data} already contains \code{time}, \code{delta} is ignored when \code{length(delta)==1}
and is an error when \code{length(delta)>1}.}

\item{parallel}{\code{TRUE} or \code{FALSE} specifying furrr/future to parallelize either:
(a) over candidate \code{delta} values (when \code{length(delta) > 1}), and/or
(b) over local multi-start initializations (when \code{starts$local > 1}), and/or
(c) over global restarts (when \code{starts$global > 1}).
For small problems, parallel overhead may outweigh speed gains.}

\item{num_cores}{number of workers to use when \code{parallel=TRUE} and
\code{set_future_plan=TRUE}.}

\item{set_future_plan}{\code{TRUE} or \code{FALSE}. If \code{TRUE} and
\code{parallel=TRUE}, set a temporary \pkg{future} plan internally and
restore the previous plan on exit.}

\item{strategy}{character string specifying the estimation strategy:
\itemize{
  \item \code{"local"}: local optimization only (single-level or multi-level polish).
  \item \code{"global_local"}: global optimization then local polish (single grid level).
  \item \code{"multires_global_local"}: multi-resolution (coarsest uses global+local; refinements use local only).
}}

\item{global_algorithm, local_algorithm}{NLopt algorithms to use for
the global and local optimization stages, respectively.}

\item{starts}{a list controlling restart and jitter behavior:
\itemize{
  \item \code{global}: integer, number of global restarts at the first/coarsest level (default 1).
  \item \code{local}: integer, number of local multi-starts per level (default 1).
  \item \code{jitter_sd}: numeric SD for jittering (default 0.35).
  \item \code{seed}: integer base seed (default 1).
}}

\item{finite_bounds}{(optional) list with components \code{lb} and \code{ub} giving finite lower and
upper bounds for all 8 parameters. If \code{NULL}, bounds are derived from \code{parameter_inits}.
Global algorithms and select local algorithms in NLopt require finite bounds.}

\item{refine_best_delta}{\code{TRUE} or \code{FALSE}. If \code{TRUE} and \code{length(delta) > 1}, performs refinement
of the best \code{delta} across additional grid levels (if available).}

\item{rescore_control}{controls candidate rescoring and bound-handling behavior in
  multi-resolution fitting. Can be either:
  
  \itemize{
    \item a single logical value (toggle rescoring on/off while keeping defaults), or
    \item a named list with any of:
      \code{enabled}, \code{top}, \code{objective_tol}, \code{param_tol},
      \code{avoid_bound_solutions}, \code{bound_eps}.
  }

  Defaults are:
  \code{list(enabled = TRUE, top = 5L, objective_tol = 1e-6, param_tol = 0.10,}
  \code{avoid_bound_solutions = TRUE, bound_eps = 1e-8)}.}

\item{verbose}{\code{TRUE} or \code{FALSE} indicating whether to show progress of model estimation.}
}
\value{
an object of class \code{"ldmppr_fit"} containing the best \code{nloptr} fit and
  (optionally) stored fits from global restarts and/or a delta search.
}
\description{
Estimate spatio-temporal point process parameters by maximizing the (approximate)
full log-likelihood using \code{\link[nloptr:nloptr]{nloptr}}.
}
\details{
For the self-correcting process, arrival times must lie on \eqn{(0,1)} and can be
supplied directly in \code{data} as \code{time}, or constructed from \code{size}
via the gentle-decay (power-law) mapping \code{\link{power_law_mapping}} using
\code{delta}. When \code{delta} is a vector, the model is fit for each candidate
value and the best objective is selected.

This function supports multi-resolution estimation via a \code{\link{ldmppr_grids}}
schedule. If multiple grid levels are provided, the coarsest level may use a global
optimizer followed by local refinement, and subsequent levels run local refinement only.
}
\examples{
# Load example data
data(small_example_data)

# Define grids and budgets
ub <- c(1, 25, 25)
g  <- ldmppr_grids(upper_bounds = ub, levels = list(c(10,10,10)))
b  <- ldmppr_budgets(
  global_options = list(maxeval = 150),
  local_budget_first_level = list(maxeval = 50, xtol_rel = 1e-2),
  local_budget_refinement_levels = list(maxeval = 25, xtol_rel = 1e-2)
)

# Estimate parameters using a single delta value
fit <- estimate_process_parameters(
  data = small_example_data,
  grids = g,
  budgets = b,
  delta = 1,
  strategy = "global_local",
  global_algorithm = "NLOPT_GN_CRS2_LM",
  local_algorithm  = "NLOPT_LN_BOBYQA",
  starts = list(global = 2, local = 2, jitter_sd = 0.25, seed = 1),
  verbose = TRUE
)
coef(fit)
logLik(fit)

\donttest{
# Estimate parameters using multiple delta values (delta search)
g2 <- ldmppr_grids(upper_bounds = ub, levels = list(c(8,8,8), c(12,12,12)))
fit_delta <- estimate_process_parameters(
  data = small_example_data, # x,y,size
  grids = g2,
  budgets = b,
  delta = c(0.35, 0.5, 0.65, 0.9, 1.0),
  parallel = TRUE,
  set_future_plan = TRUE,
  num_cores = 2,
  strategy = "multires_global_local",
  starts = list(local = 1),
  refine_best_delta = FALSE,
  verbose = FALSE
)
plot(fit_delta)
}

}
\references{
Møller, J., Ghorbani, M., & Rubak, E. (2016). Mechanistic spatio-temporal point process models
for marked point processes, with a view to forest stand data. \emph{Biometrics}, 72(3), 687-696.
\doi{10.1111/biom.12466}.
}
