% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kindling-package.R
\docType{package}
\name{kindling}
\alias{kindling-package}
\alias{kindling}
\title{\code{{kindling}}: Higher-level interface of torch package to auto-train neural networks}
\description{
\code{{kindling}} enables R users to build and train deep neural networks such as:
\itemize{
\item Deep Neural Networks / (Deep) Feedforward Neural Networks (DNN / FFNN)
\item Recurrent Neural Networks (RNN)
}

It is designed to reduce boilerplate \code{{torch}} code for FFNN and RNN. It also
integrate seamlessly with \code{{tidymodels}} components like \code{{parsnip}}, \code{{recipes}},
and \code{{workflows}}, allowing flexibility and a consistent interface for model
specification, training, and evaluation.

Thus, the package supports hyperparameter tuning for:
\itemize{
\item Number of hidden layers
\item Number of units per layer
\item Choice of activation functions
}

Note: The hyperparameter tuning support is not currently implemented.
}
\details{
The \code{{kindling}} package provides a unified, high-level interface that bridges
the \code{{torch}} and \code{{tidymodels}} ecosystems, making it easy to define, train,
and tune deep learning models using the familiar \code{tidymodels} workflow.
}
\section{How to use}{

The following uses of this package has 3 levels:

Level 1: Code generation

\if{html}{\out{<div class="sourceCode r">}}\preformatted{ffnn_generator(
    nn_name = "MyFFNN",
    hd_neurons = c(64, 32, 16),
    no_x = 10,
    no_y = 1,
    activations = 'relu'
)
}\if{html}{\out{</div>}}

Level 2: Direct Execution

\if{html}{\out{<div class="sourceCode r">}}\preformatted{ffnn(
    Species ~ .,
    data = iris,
    hidden_neurons = c(128, 64, 32),
    activations = 'relu',
    loss = "cross_entropy",
    epochs = 100
)
}\if{html}{\out{</div>}}

Level 3: Conventional tidymodels interface

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# library(parsnip)
# library(kindling)
box::use(
   kindling[mlp_kindling, rnn_kindling, act_funs, args],
   parsnip[fit, augment],
   yardstick[metrics],
   mlbench[Ionosphere] # data(Ionosphere, package = "mlbench")
)

# Remove V2 as it's all zeros
ionosphere_data = Ionosphere[, -2]

# MLP example
mlp_kindling(
    mode = "classification",
    hidden_neurons = c(128, 64),
    activations = act_funs(relu, softshrink = args(lambd = 0.5)),
    epochs = 100
) |>
    fit(Class ~ ., data = ionosphere_data) |>
    augment(new_data = ionosphere_data) |>
    metrics(truth = Class, estimate = .pred_class)
#> A tibble: 2 × 3
#>   .metric  .estimator .estimate
#>   <chr>    <chr>          <dbl>
#> 1 accuracy binary         0.989
#> 2 kap      binary         0.975

# RNN example (toy usage on non-sequential data)
rnn_kindling(
    mode = "classification",
    hidden_neurons = c(128, 64),
    activations = act_funs(relu, elu),
    epochs = 100,
    rnn_type = "gru"
) |>
    fit(Class ~ ., data = ionosphere_data) |>
    augment(new_data = ionosphere_data) |>
    metrics(truth = Class, estimate = .pred_class)
#> A tibble: 2 × 3
#>   .metric  .estimator .estimate
#>   <chr>    <chr>          <dbl>
#> 1 accuracy binary         0.641
#> 2 kap      binary         0
}\if{html}{\out{</div>}}
}

\section{Main Features}{

\itemize{
\item Code generation of \code{{torch}} expression
\item Multiple architectures available: feedforward networks (MLP/DNN/FFNN) and recurrent variants (RNN, LSTM, GRU)
\item Native support for \code{{tidymodels}} workflows and pipelines
\item Fine-grained control over network depth, layer sizes, and activation functions
\item GPU acceleration supports via \code{{torch}} tensors
}
}

\section{License}{

MIT + file LICENSE
}

\references{
Falbel D, Luraschi J (2023). \emph{torch: Tensors and Neural Networks with 'GPU'
Acceleration}. R package version 0.13.0,
\url{https://torch.mlverse.org}, \url{https://github.com/mlverse/torch}.

Wickham H (2019). \emph{Advanced R}, 2nd edition. Chapman and Hall/CRC.
ISBN 978-0815384571, \url{https://adv-r.hadley.nz/}.

Goodfellow I, Bengio Y, Courville A (2016). \emph{Deep Learning}. MIT Press.
\url{https://www.deeplearningbook.org/}.
}
\seealso{
Useful links:
\itemize{
  \item \url{https://kindling.joshuamarie.com}
  \item \url{https://github.com/joshuamarie/kindling}
  \item Report bugs at \url{https://github.com/joshuamarie/kindling/issues}
}

}
\author{
\strong{Maintainer}: Joshua Marie \email{joshua.marie.k@gmail.com}

}
\keyword{internal}
