% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/jmSurf.R
\name{jmSurf}
\alias{jmSurf}
\title{Fit a Joint Longitudinal-Survival Model with Semi-Parametric Association Surfaces}
\usage{
jmSurf(
  long_data,
  surv_data,
  transitions = NULL,
  covariates = c("age_baseline", "sex"),
  k_marginal = c(5, 5),
  k_additive = 6,
  bs = "tp",
  method = "REML",
  min_events = 10,
  verbose = TRUE
)
}
\arguments{
\item{long_data}{Data frame of longitudinal biomarker measurements.
Required columns: \code{patient_id}, \code{visit_time_years},
\code{biomarker}, \code{value}.}

\item{surv_data}{Data frame of survival/transition events.
Required columns: \code{patient_id}, \code{start_time}, \code{stop_time},
\code{status}, \code{state_from}, \code{state_to}, \code{transition}.}

\item{transitions}{Character vector of transitions to model (e.g.,
\code{"CKD -> CVD"}). If \code{NULL}, all observed transitions are used.}

\item{covariates}{Character vector of baseline covariate names present in
\code{surv_data}. Default \code{c("age_baseline", "sex")}.}

\item{k_marginal}{Integer vector of length 1 or 2 giving marginal basis
dimensions for the tensor-product spline. Default \code{c(5, 5)}.}

\item{k_additive}{Integer giving the basis dimension for the additive
smooth of the third biomarker (if present). Default \code{6}.}

\item{bs}{Character string for the spline basis type. Default \code{"tp"}
(thin-plate regression spline).}

\item{method}{Smoothing parameter estimation method. Default \code{"REML"}.}

\item{min_events}{Integer minimum number of events required to fit a
transition model. Default \code{10}.}

\item{verbose}{Logical; print progress messages. Default \code{TRUE}.}
}
\value{
An object of class \code{"jmSurface"} containing:
\item{lme_fits}{Named list of \code{nlme::lme} objects (one per biomarker)}
\item{gam_fits}{Named list of \code{mgcv::gam} objects (one per transition)}
\item{eta_data}{Named list of analysis data frames with latent summaries}
\item{transitions}{Character vector of fitted transitions}
\item{biomarkers}{Character vector of biomarker names}
\item{covariates}{Character vector of covariate names used}
\item{edf}{Named numeric vector of EDF values per transition}
\item{deviance_explained}{Named numeric of deviance explained per transition}
\item{call}{The matched call}
}
\description{
Two-stage estimation framework for multi-state joint models with
tensor-product spline association surfaces. Stage 1 fits mixed-effects
longitudinal models and extracts BLUPs. Stage 2 fits transition-specific
penalized Cox models with tensor-product spline surfaces via REML.
}
\details{
The model for each transition \eqn{(r,s)} is:
\deqn{\lambda_i^{rs}(t | \eta_i(t)) = \lambda_0^{rs}(t) \exp\{\gamma_{rs}' w_i + f_{rs}(\eta_i(t))\}}
where \eqn{f_{rs}} is a semi-parametric association surface represented via
tensor-product splines, and \eqn{\eta_i(t)} are BLUP-based latent
longitudinal summaries evaluated at the midpoint of each sojourn interval.
}
\examples{
\donttest{
# Simulate data
sim <- simulate_jmSurface(n_patients = 300)

# Fit the joint model
fit <- jmSurf(
  long_data = sim$long_data,
  surv_data = sim$surv_data,
  covariates = c("age_baseline", "sex")
)

# Summary with EDF diagnostics
summary(fit)

# Dynamic prediction for patient 1
pred <- dynPred(fit, patient_id = 1, landmark = 2, horizon = 3)

# Visualize surfaces
plot_surface(fit, transition = "CKD -> CVD")
contour_heatmap(fit, transition = "CKD -> CVD")
marginal_slices(fit, transition = "CKD -> CVD")
}

}
\references{
Bhattacharjee, A. (2025). Interpretable Multi-Biomarker Fusion in Joint
Longitudinal-Survival Models via Semi-Parametric Association Surfaces.

Bhattacharjee, A. (2024). jmBIG: Scalable Joint Models for Big Data.

Wood, S.N. (2017). Generalized Additive Models: An Introduction with R.
Chapman & Hall/CRC.

Tsiatis, A.A. & Davidian, M. (2004). Joint modeling of longitudinal and
time-to-event data: an overview. Statistica Sinica, 14, 809-834.
}
