#' @title Bias-Corrected Meta-Analysis for Combining Studies of Different Types and Quality
#'
#' @description This function performs a Bayesian meta-analysis to jointly
#' combine different types of studies. The random-effects follows a finite
#' mixture of normal distributions.
#'
#'
#'
#' @param data                A data frame with at least two columns with the following names:
#'                             1) TE = treatment effect,
#'                             2) seTE = the standard error of the treatment effect.
#'
#' @param mean.mu.theta             Prior mean of the overall mean parameter mu, default value is 0.
#'
#' @param sd.mu.theta               Prior standard deviation of mu, the default value is 10.
#'
#' @param scale.sigma.between Prior scale parameter for scale gamma distribution for the
#'                            precision between studies. The default value is 0.5.
#'
#' @param df.scale.between    Degrees of freedom of the scale gamma distribution for the precision between studies.
#'                            The default value is 1, which results in a Half Cauchy distribution for the standard
#'                            deviation between studies. Larger values e.g. 30 corresponds to a Half Normal distribution.
#'
#' @param B.lower             Lower bound of the bias parameter B, the default value is 0.
#' @param B.upper             Upper bound of the bias parameter B, the default value is 10.
#'
#' @param a.0                 Parameter for the prior Beta distribution for the probability of bias. Default value is a0 = 1.
#' @param a.1                 Parameter for the prior Beta distribution for the probability of bias. Default value is a1 = 1.
#'
#' @param nu                  Parameter for the Beta distribution for the quality weights. The default value is nu = 0.5.
#'
#' @param nu.estimate         If TRUE, then we estimate nu from the data.
#'
#' @param b.0                 If nu.estimate = TRUE, this parameter is the shape parameter of the prior Gamma distribution for nu.
#' @param b.1                 If nu.estimate = TRUE, this parameter is the rate parameter of the prior Gamma distribution for nu.
#'                            Note that E(nu) = b.0/b.1 and we need to choose b.0 << b.1.
#'
#' @param preclassification A character string indicating how to pre-classify the study with the most extreme effect size
#'                           as biased. Options are: \code{"min"} (smallest effect is pre-classified as **unbiased, I=0**),
#'                           \code{"max"} (largest effect is pre-classified as **biased, I=1** - default),
#'                           \code{"both"} (min as **I=0** and max as **I=1**), or \code{NULL} (no pre-classification).
#'                           **This argument is ignored if `inits` is provided.**
#'
#' @param inits               Optional initial values for MCMC chains, passed to \code{jags} or \code{jags.parallel}. Default is \code{NULL}.
#'
#' @param nr.chains           Number of chains for the MCMC computations, default 2.
#' @param nr.iterations       Number of iterations after adapting the MCMC, default is 10000. Some models may need more iterations.
#' @param nr.adapt            Number of iterations in the adaptation process, defualt is 1000. Some models may need more iterations during adptation.
#' @param nr.burnin           Number of iteration discared for burnin period, default is 1000. Some models may need a longer burnin period.
#' @param nr.thin             Thinning rate, it must be a positive integer, the default value 1.
#' @param parallel            NULL -> jags, 'jags.parallel' -> jags.parallel execution
#'
#' @return                    This function returns an object of the class "bcmeta". This object contains the MCMC
#'                            output of each parameter and hyper-parameter in the model and
#'                            the data frame used for fitting the model.
#'
#'
#' @details The results of the object of the class bcmeta can be extracted with R2jags or with rjags. In addition a summary, a print and a plot functions are
#' implemented for this type of object.
#'
#'
#' @references Verde, P. E. (2017) Two Examples of Bayesian Evidence Synthesis with the Hierarchical Meta-Regression Approach. Chap.9, pag 189-206. Bayesian Inference, ed. Tejedor, Javier Prieto. InTech.
#'
#' @references Verde, P.E. (2021) A Bias-Corrected Meta-Analysis Model for Combining Studies of Different Types and Quality. Biometrical Journal; 1–17.
#'
#'
#' @examples
#' \dontrun{
#' library(jarbes)
#'
#' # Example ppvipd data
#'
#' data(ppvipd)
#'
# bcm1 = bcmeta(ppvipd, a.0 = 8, a.1 = 3) # 8 OS and 3 RCT...
# summary(bcm1)
# plot(bcm1)
#
# diagnostic(bcm1, study.names = ppvipd$name,
#            post.p.value.cut = 0.1, shape.forest = 4, lwd.forest = 0.75)
#
#
# diagnostic(bcm1, post.p.value.cut = 0.1,
#            shape.forest = 4, lwd.forest = 0.75,
#            y.lim = c(0, 3),
#            color.data.points = "blue",
#            bias.plot = TRUE,
#            S = 10000)
#
# # Example with stem cells
#
# data("stemcells")
# stemcells$TE = stemcells$effect.size
# stemcells$seTE = stemcells$se.effect
#
# bcm2 = bcmeta(stemcells, mean.mu = 0, sd.mu = 1000,
#               nr.iterations = 50000,
#               nr.adapt = 20000,
#               nr.thin = 2)
#
# summary(bcm2)
#
# plot(bcm2, x.lim = c(-5, 15), y.lim = c(0, .8))
#
# diagnostic(bcm2, study.names = stemcells$trial,
#            post.p.value.cut = 0.05,
#            shape.forest = 4, lwd.forest = 0.75)
#
# # only bias plot
# diagnostic(bcm2, study.names = stemcells$trial,
#            post.p.value.cut = 0.05,
#            cross.val.plot = FALSE,
#            shape.forest = 4, lwd.forest = 0.75)
#
# # only CV plot
# diagnostic(bcm2, study.names = stemcells$trial,
#            post.p.value.cut = 0.05,
#            cross.val.plot = TRUE,
#            bias.plot = FALSE,
#            shape.forest = 4, lwd.forest = 0.75)
#
#'
#' }
#'
#' @import R2jags
#' @import rjags
#'
#'
#' @export
bcmeta = function(
    data,
    # Hyperpriors parameters............................................
    mean.mu.theta     = 0,
    sd.mu.theta       = 10,
    scale.sigma.between = 0.5,
    df.scale.between    = 1,
    B.lower     = 0,
    B.upper     = 10,

    a.0         = 1,
    a.1         = 1,
    nu          = 0.5,
    nu.estimate = FALSE,
    b.0 = 1,
    b.1 = 2,

    # New arguments...................................................
    preclassification = "max",
    inits = NULL,

    # MCMC setup........................................................
    nr.chains       = 2,
    nr.iterations   = 10000,
    nr.adapt        = 1000,
    nr.burnin       = 1000,
    nr.thin         = 1,
    parallel        = NULL
)UseMethod("bcmeta")



#' @export
bcmeta.default = function(
    data,
    # Hyperpriors parameters............................................
    mean.mu.theta     = 0,
    sd.mu.theta       = 10,
    scale.sigma.between = 0.5,
    df.scale.between    = 1,
    B.lower     = 0,
    B.upper     = 10,
    a.0         = 1,
    a.1         = 1,
    nu          = 0.5,
    nu.estimate = FALSE,
    b.0         = 1,
    b.1         = 2,

    # New arguments...................................................
    preclassification = "max",
    inits = NULL,

    # MCMC setup........................................................
    nr.chains       = 2,
    nr.iterations   = 10000,
    nr.adapt        = 1000,
    nr.burnin       = 1000,
    nr.thin         = 1,
    parallel        = NULL)
{
  if(!is.null(parallel) && parallel != "jags.parallel") stop("The parallel option must be NULL or 'jags.parallel'")


  # --- Outcome data with NA trick for prediction ------------------------
  y <- c(data$TE, NA)
  se.y <- c(data$seTE, 1)
  N <- length(y)

  if(N<5)stop("Low number of studies in the meta-analysis!")

  # -----------------------------------------------------------------
  # CONDITIONAL PRE-CLASSIFICATION / INITIALIZATION OF I
  # Do NOT initialize I if the user provides custom initial values ('inits').
  I = NULL # Placeholder: I is only calculated below if inits is NULL

  if (is.null(inits)) {
    I = rep(NA, N) # I is initialized to NA for estimation, or 0/1 if preclassified.

    # PRE-CLASSIFICATION LOGIC
    if (!is.null(preclassification)) {
      min.index = which(y == min(y, na.rm = TRUE))
      max.index = which(y == max(y, na.rm = TRUE))

      # Convert to lower case for robust checking
      preclassification_lc <- tolower(preclassification)

      if (preclassification_lc == "min") {
        # Minimum effect is pre-classified as UNBIASED (I=0)
        I[min.index] = 0
      } else if (preclassification_lc == "max") {
        # Maximum effect is pre-classified as BIASED (I=1)
        I[max.index] = 1
      } else if (preclassification_lc == "both") {
        # Minimum effect is pre-classified as UNBIASED (I=0)
        I[min.index] = 0
        # Maximum effect is pre-classified as BIASED (I=1)
        I[max.index] = 1
      } else {
        warning("Invalid value for preclassification. Must be 'min', 'max', 'both', or NULL. Defaulting to 'max'.")
        I[max.index] = 1
      }
    }
  }
  # -----------------------------------------------------------------


  # This list describes the data used by the BUGS script.
  data.bcmeta <-
    list(y = y,
         se.y = se.y,
         N = N,
         mean.mu.theta = mean.mu.theta,
         sd.mu.theta = sd.mu.theta,
         scale.sigma.between = scale.sigma.between,
         df.scale.between = df.scale.between,
         B.lower = B.lower,
         B.upper = B.upper,
         a.0 = a.0,
         a.1 = a.1,
         nu = nu
    )

  if (is.null(inits)) {
    # I is added to the data only if it was initialized by the function (i.e., if inits is NULL).
    data.bcmeta$I = I
    # Reorder elements to ensure I is in the expected position (position 4)
    data.bcmeta <- data.bcmeta[c("y", "se.y", "N", "I",
                                 "mean.mu.theta", "sd.mu.theta", "scale.sigma.between",
                                 "df.scale.between", "B.lower", "B.upper", "a.0", "a.1", "nu")]
  }


  if(nu.estimate==TRUE) {
    data.bcmeta = data.bcmeta[names(data.bcmeta) != "nu"]    # Remove "nu" by name for robustness
    data.bcmeta = c(data.bcmeta, b.0 = b.0, b.1 = b.1)       # Add hyper constants for "nu
  }

  # List of parameters
  par.bcmeta <- c("mu.theta",
                  "sd.theta",
                  "theta.bias",
                  "theta",
                  "beta",
                  "I",
                  "pi.B",
                  "q",
                  "mu.beta")

  if(nu.estimate==TRUE) {par.bcmeta = c(par.bcmeta, "nu" )}

  # nu ~ dgamma(b.0, b.1)

  # Model in BUGS

  model.bugs <-
    "
  model
  {
   for( i in 1 : N ) {

  # Likelihood of theta.bias[i] ................................................
            y[i] ~ dnorm(theta.bias[i], 1 / se.y[i]^2)

 # Mixture Process: Normal and Normal+bias       ..............................

  theta.bias[i] <- theta[i] + (beta[i]*I[i])
           I[i] ~ dbern(pi.B)
       theta[i] ~ dnorm(mu.theta, inv.var.theta)
        beta[i] ~ dnorm(mu.beta,  inv.var.theta * q[i]/(1-q[i]))
           q[i] ~ dbeta(nu, 1)

  # q[i] = tau_theta^2 /( tau_theta^2 + tau_beta_i^2) see the paper........
}

# Priors for hyper-parameters .................................................

# Prior for the probability of bias.........................................
     pi.B ~ dbeta(a.0, a.1)

# Priors for model of interest ..............................................

      mu.theta ~ dnorm(mean.mu.theta, inv.var.mu.theta)
      inv.var.mu.theta <- pow(sd.mu.theta, -2)

      inv.var.theta ~ dscaled.gamma(scale.sigma.between, df.scale.between)
          sd.theta <- pow(inv.var.theta, -0.5)

# Prior for the mean bias ....................................................

     mu.beta ~ dunif(B.lower, B.upper)

  }
"

if (is.null(parallel)) { #execute R2jags
  model.bugs.connection <- textConnection(model.bugs)

  # Use R2jags as interface for JAGS ...

  results <- jags( data = data.bcmeta,
                   parameters.to.save = par.bcmeta,
                   model.file = model.bugs.connection,
                   n.chains = nr.chains,
                   n.iter = nr.iterations,
                   n.burnin = nr.burnin,
                   n.thin = nr.thin,
                   inits = inits,
                   DIC = TRUE,
                   pD=TRUE)

  # Close text connection
  close(model.bugs.connection)
}else if(parallel == "jags.parallel"){
  writeLines(model.bugs, "model.bugs")
  results <- jags.parallel(     data = data.bcmeta,
                                parameters.to.save = par.bcmeta,
                                model.file = "model.bugs",
                                n.chains = nr.chains,
                                n.iter = nr.iterations,
                                n.burnin = nr.burnin,
                                n.thin = nr.thin,
                                inits = inits,
                                DIC=TRUE)

  #Compute pD from result
  results$BUGSoutput$pD = results$BUGSoutput$DIC - results$BUGSoutput$mean$deviance

  # Delete model.bugs on exit ...
  unlink("model.bugs")
}


# Extra outputs that are linked with other functions ...
results$data = data

# Hyperpriors parameters............................................
results$prior$mean.mu     = mean.mu.theta
results$prior$sd.mu       = sd.mu.theta
results$prior$scale.sigma.between = scale.sigma.between
results$prior$df.scale.between    = df.scale.between
results$prior$B.lower     = B.lower
results$prior$B.upper     = B.upper

results$prior$a.0         = a.0
results$prior$a.1         = a.1
results$prior$nu          = nu
results$prior$nu.estimate = nu.estimate
results$prior$b.0 = b.0
results$prior$b.1 = b.1

class(results) = c("bcmeta")

return(results)
}



#' Generic print function for bcmeta object in jarbes.
#'
#' @param x The object generated by the function bcmeta.
#'
#' @param digits The number of significant digits printed. The default value is 3.
#'
#' @param ... \dots
#'
#' @export
print.bcmeta <- function(x, digits, ...)
{
  print(x$BUGSoutput,...)
}
