% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/igroup-process.R
\name{igroup_process}
\alias{igroup_process}
\title{Handle unexpected additional grouping structure}
\usage{
igroup_process(df = NULL, fn, ..., .iface = NULL)
}
\arguments{
\item{df}{a dataframe from an enclosing function in which the grouping may or
may not have been correctly supplied.}

\item{fn}{a function to call with the correctly grouped dataframe as
specified by the \code{iface} in the enclosing function. This must be a function
(not a \code{purrr} style lambda) and share parameter naming with the enclosing
function. The first parameter of this dispatch function must be the
dataframe (correctly grouped), and other named parameters here are looked
for in the enclosing function call. The function \emph{must} return a dataframe.}

\item{...}{passed onto \code{iconvert} this could be used to supply
\code{.prune} parameters. triple dot parameters in the enclosing function will
be separately handled and automatically passed to \code{fn} so in general should
not be passed to \code{igroup_process} as an intermediary although it probably
won't hurt. This behaviour is similar to \code{NextMethod} in S3 method
dispatch.}

\item{.iface}{experimental feature to override the detection of the data from
the enclosing function and use a directly supplied one. This can be useful
if you want to process a dataframe by grouping defined by some criteria
supplied at runtime.}
}
\value{
the result of calling \code{fn(df, ...)} on each unexpected group
}
\description{
This function is designed to be used by a package author within an enclosing
function. The enclosing function is assumed to take as input a dataframe and
have an \code{iface} specified for that dataframe.
}
\details{
This function detects when the grouping of the input has additional groups
over and above those in the specification and intercepts them, regrouping the
dataframe and applying \code{fn} group-wise using an equivalent of a
\code{dplyr::group_modify}. The parameters provided to the enclosing function will
be passed to \code{fn} and they should have compatible method signatures.
}
\examples{

# This specification requires that the dataframe is grouped only by the color
# column
i_diamond_price = interfacer::iface(
  color = enum(`D`,`E`,`F`,`G`,`H`,`I`,`J`, .ordered=TRUE) ~ "the color column",
  price = integer ~ "the price column",
  .groups = ~ color
)

# An example function which would be exported in a package
# This function expects a dataframe with a colour and price column, grouped
# by price.
mean_price_by_colour = function(df = i_diamond_price, extra_param = ".") {

  # When called with a dataframe with extra groups `igroup_process` will
  # regroup the dataframe according to the structure
  # defined for `i_diamond_price` and apply the inner function to each group
  # after first calling `ivalidate` on each group.

  igroup_process(df,

    # the real work of this function is provided as an anonymous inner
    # function (but can be any other function e.g. package private function
    # but not a purrr style lambda). Ideally this function parameters are named the
    # same as the enclosing function (here `ex_mean(df,extra_param)`), however
    # there is some flexibility here. The special `.groupdata` parameter will
    # be populated with the values of the unexpected grouping.

    function(df, extra_param, .groupdata) {
      message(extra_param, appendLF = FALSE)
      if (nrow(.groupdata) == 0) message("zero length group data")
      return(df \%>\% dplyr::summarise(mean_price = mean(price)))
    }

  )
}

# The correctly grouped dataframe. The `ex_mean` function calculates the mean
# price for each `color` group.
ggplot2::diamonds \%>\%
  dplyr::group_by(color) \%>\%
  mean_price_by_colour(extra_param = "without additional groups...") \%>\%
  dplyr::glimpse()

# If an additionally grouped dataframe is provided by the user. The `ex_mean`
# function calculates the mean price for each `cut`,`clarity`, and `color`
# combination.

ggplot2::diamonds \%>\%
  dplyr::group_by(cut, color, clarity) \%>\%
  mean_price_by_colour() \%>\%
  dplyr::glimpse()

# The output of this is actually grouped by cut then clarity as
# color is consumed by the `igroup_dispatch`.

# This example is somewhat contorted. The real power of `igroup_process` is
# if it is used recursively:

recursive_example = function(df = i_diamond_price) {

  # call main function recursively if additional groups detected
  igroup_process(df, recursive_example)
  # N.B. this also works if the second argument is omitted e.g.:
  # igroup_process(df)

  # otherwise proceed with function as normal
  return(tibble::tibble("rows detected:"=nrow(df)))
}

ggplot2::diamonds \%>\% dplyr::group_by(color) \%>\%
   recursive_example() \%>\%
   dplyr::glimpse()
ggplot2::diamonds \%>\% dplyr::group_by(cut,clarity,color) \%>\%
   recursive_example() \%>\%
   dplyr::glimpse()
}
\concept{interface}
