/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Fmodule.h" 
#define H5F_TESTING    
#define H5G_FRIEND     
#define H5G_TESTING    
#define H5SM_FRIEND    
#define H5SM_TESTING   

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5Gpkg.h"      
#include "H5Iprivate.h"  
#include "H5SMpkg.h"     
#include "H5VLprivate.h" 

herr_t
H5F__get_sohm_mesg_count_test(hid_t file_id, unsigned type_id, size_t *mesg_count)
{
    H5F_t      *file;                         
    H5CX_node_t api_ctx        = {{0}, NULL}; 
    bool        api_ctx_pushed = false;       
    herr_t      ret_value      = SUCCEED;     

    FUNC_ENTER_PACKAGE

    
    if (NULL == (file = (H5F_t *)H5VL_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file");

    
    if (H5CX_push(&api_ctx) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "can't set API context");
    api_ctx_pushed = true;

    
    if (H5SM__get_mesg_count_test(file, type_id, mesg_count) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't retrieve shared message count");

done:
    if (api_ctx_pushed && H5CX_pop(false) < 0)
        HDONE_ERROR(H5E_FILE, H5E_CANTRESET, FAIL, "can't reset API context");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__check_cached_stab_test(hid_t file_id)
{
    H5F_t      *file;                         
    H5CX_node_t api_ctx        = {{0}, NULL}; 
    bool        api_ctx_pushed = false;       
    herr_t      ret_value      = SUCCEED;     

    FUNC_ENTER_PACKAGE

    
    if (NULL == (file = (H5F_t *)H5VL_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file");

    
    if (H5CX_push(&api_ctx) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "can't set API context");
    api_ctx_pushed = true;

    
    if (H5G__verify_cached_stab_test(H5G_oloc(file->shared->root_grp), file->shared->sblock->root_ent) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to verify cached symbol table info");

done:
    if (api_ctx_pushed && H5CX_pop(false) < 0)
        HDONE_ERROR(H5E_FILE, H5E_CANTRESET, FAIL, "can't reset API context");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__get_maxaddr_test(hid_t file_id, haddr_t *maxaddr)
{
    H5F_t *file;                
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (file = (H5F_t *)H5VL_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file");

    
    *maxaddr = file->shared->maxaddr;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__get_sbe_addr_test(hid_t file_id, haddr_t *sbe_addr)
{
    H5F_t *file;                
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (file = (H5F_t *)H5VL_object_verify(file_id, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file");

    
    *sbe_addr = file->shared->sblock->ext_addr;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5F__same_file_test(hid_t file_id1, hid_t file_id2)
{
    H5F_t *file1, *file2;    
    htri_t ret_value = FAIL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (file1 = (H5F_t *)H5VL_object_verify(file_id1, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file");
    if (NULL == (file2 = (H5F_t *)H5VL_object_verify(file_id2, H5I_FILE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file");

    
    ret_value = (file1->shared == file2->shared);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__reparse_file_lock_variable_test(void)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    if (H5F__parse_file_lock_env_var(&use_locks_env_g, &ignore_disabled_locks_g) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "unable to parse file locking environment variable");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
