/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Fmodule.h" 
#define H5G_FRIEND     

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Fpkg.h"     
#include "H5Gpkg.h"     
#include "H5Iprivate.h" 
#include "H5Pprivate.h" 

herr_t
H5F_debug(H5F_t *f, FILE *stream, int indent, int fwidth)
{
    H5P_genplist_t *plist;               
    hsize_t         userblock_size;      
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(f->shared->fcpl_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");

    
    if (H5P_get(plist, H5F_CRT_USER_BLOCK_NAME, &userblock_size) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get userblock size");

    
    Rfprintf(stream, "%*sFile Super Block...\n", indent, "");

    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "File name (as opened):", H5F_OPEN_NAME(f));
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth,
            "File name (after resolving symlinks):", H5F_ACTUAL_NAME(f));
    Rfprintf(stream, "%*s%-*s 0x%08x\n", indent, "", fwidth, "File access flags", f->shared->flags);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "File open reference count:", f->shared->nrefs);
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR " (abs)\n", indent, "", fwidth,
            "Address of super block:", f->shared->sblock->base_addr);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE " bytes\n", indent, "", fwidth,
            "Size of userblock:", userblock_size);

    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Superblock version number:", f->shared->sblock->super_vers);

    
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Free list version number:", (unsigned)HDF5_FREESPACE_VERSION);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Root group symbol table entry version number:", (unsigned)HDF5_OBJECTDIR_VERSION);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Shared header version number:", (unsigned)HDF5_SHAREDHEADER_VERSION);

    Rfprintf(stream, "%*s%-*s %u bytes\n", indent, "", fwidth,
            "Size of file offsets (haddr_t type):", (unsigned)f->shared->sizeof_addr);
    Rfprintf(stream, "%*s%-*s %u bytes\n", indent, "", fwidth,
            "Size of file lengths (hsize_t type):", (unsigned)f->shared->sizeof_size);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Symbol table leaf node 1/2 rank:", f->shared->sblock->sym_leaf_k);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Symbol table internal node 1/2 rank:", f->shared->sblock->btree_k[H5B_SNODE_ID]);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Indexed storage internal node 1/2 rank:", f->shared->sblock->btree_k[H5B_CHUNK_ID]);
    Rfprintf(stream, "%*s%-*s 0x%02x\n", indent, "", fwidth,
            "File status flags:", (unsigned)(f->shared->sblock->status_flags));
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR " (rel)\n", indent, "", fwidth,
            "Superblock extension address:", f->shared->sblock->ext_addr);
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR " (rel)\n", indent, "", fwidth,
            "Shared object header message table address:", f->shared->sohm_addr);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Shared object header message version number:", (unsigned)f->shared->sohm_vers);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Number of shared object header message indexes:", (unsigned)f->shared->sohm_nindexes);

    Rfprintf(stream, "%*s%-*s %" PRIuHADDR " (rel)\n", indent, "", fwidth,
            "Address of driver information block:", f->shared->sblock->driver_addr);

    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth,
            "Root group symbol table entry:", f->shared->root_grp ? "" : "(none)");
    if (f->shared->root_grp) {
        if (f->shared->sblock->root_ent) 
            H5G__ent_debug(f->shared->sblock->root_ent, stream, indent + 3, MAX(0, fwidth - 3), NULL);
        else {
            H5O_loc_t  *root_oloc; 
            H5G_entry_t root_ent;  

            
            H5G__ent_reset(&root_ent);

            
            root_oloc = H5G_oloc(f->shared->root_grp);
            assert(root_oloc);
            root_ent.type   = H5G_NOTHING_CACHED;
            root_ent.header = root_oloc->addr;

            
            H5G__ent_debug(&root_ent, stream, indent + 3, MAX(0, fwidth - 3), NULL);
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
