/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Amodule.h" 
#define H5O_FRIEND     

#include "H5private.h"   
#include "H5Apkg.h"      
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5Iprivate.h"  
#include "H5Opkg.h"      
#include "H5VLprivate.h" 

#include "H5VLnative_private.h" 

#ifndef H5_NO_DEPRECATED_SYMBOLS

hid_t
H5Acreate1(hid_t loc_id, const char *name, hid_t type_id, hid_t space_id, hid_t acpl_id)
{
    void             *attr    = NULL; 
    H5VL_object_t    *vol_obj = NULL; 
    H5VL_loc_params_t loc_params;
    hid_t             ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5I_ATTR == H5I_get_type(loc_id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "location is not valid for an attribute");
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "no name");

    
    if (H5CX_set_loc(loc_id) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTSET, H5I_INVALID_HID, "can't set collective metadata read");

    
    if (H5P_DEFAULT == acpl_id)
        acpl_id = H5P_ATTRIBUTE_CREATE_DEFAULT;

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid location identifier");

    
    if (NULL ==
        (attr = H5VL_attr_create(vol_obj, &loc_params, name, type_id, space_id, acpl_id,
                                 H5P_ATTRIBUTE_ACCESS_DEFAULT, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTINIT, H5I_INVALID_HID, "unable to create attribute");

    
    if ((ret_value = H5VL_register(H5I_ATTR, attr, H5VL_OBJ_CONNECTOR(vol_obj), true)) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register attribute for ID");

done:
    
    if (H5I_INVALID_HID == ret_value)
        if (attr && H5VL_attr_close(vol_obj, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, H5I_INVALID_HID, "can't close attribute");

    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Aopen_name(hid_t loc_id, const char *name)
{
    void             *attr    = NULL; 
    H5VL_object_t    *vol_obj = NULL; 
    H5VL_loc_params_t loc_params;
    hid_t             ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5I_ATTR == H5I_get_type(loc_id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "location is not valid for an attribute");
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "no name");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid location identifier");

    
    if (NULL == (attr = H5VL_attr_open(vol_obj, &loc_params, name, H5P_ATTRIBUTE_ACCESS_DEFAULT,
                                       H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, H5I_INVALID_HID, "unable to open attribute");

    
    if ((ret_value = H5VL_register(H5I_ATTR, attr, H5VL_OBJ_CONNECTOR(vol_obj), true)) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register attribute handle");

done:
    
    if (H5I_INVALID_HID == ret_value)
        if (attr && H5VL_attr_close(vol_obj, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, H5I_INVALID_HID, "can't close attribute");

    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Aopen_idx(hid_t loc_id, unsigned idx)
{
    void             *attr    = NULL; 
    H5VL_object_t    *vol_obj = NULL; 
    H5VL_loc_params_t loc_params;
    hid_t             ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5I_ATTR == H5I_get_type(loc_id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "location is not valid for an attribute");

    
    loc_params.type                         = H5VL_OBJECT_BY_IDX;
    loc_params.loc_data.loc_by_idx.name     = ".";
    loc_params.loc_data.loc_by_idx.idx_type = H5_INDEX_CRT_ORDER;
    loc_params.loc_data.loc_by_idx.order    = H5_ITER_INC;
    loc_params.loc_data.loc_by_idx.n        = (hsize_t)idx;
    loc_params.loc_data.loc_by_idx.lapl_id  = H5P_LINK_ACCESS_DEFAULT;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid location identifier");

    
    if (NULL == (attr = H5VL_attr_open(vol_obj, &loc_params, NULL, H5P_ATTRIBUTE_ACCESS_DEFAULT,
                                       H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, H5I_INVALID_HID, "unable to open attribute");

    
    if ((ret_value = H5VL_register(H5I_ATTR, attr, H5VL_OBJ_CONNECTOR(vol_obj), true)) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register attribute handle");

done:
    
    if (H5I_INVALID_HID == ret_value)
        if (attr && H5VL_attr_close(vol_obj, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, H5I_INVALID_HID, "can't close attribute");

    FUNC_LEAVE_API(ret_value)
} 

int
H5Aget_num_attrs(hid_t loc_id)
{
    H5VL_object_t         *vol_obj = NULL; 
    H5VL_object_get_args_t vol_cb_args;    
    H5VL_loc_params_t      loc_params;
    H5O_info2_t            oinfo;
    int                    ret_value = -1;

    FUNC_ENTER_API((-1))

    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, (-1), "invalid location identifier");

    
    vol_cb_args.op_type              = H5VL_OBJECT_GET_INFO;
    vol_cb_args.args.get_info.oinfo  = &oinfo;
    vol_cb_args.args.get_info.fields = H5O_INFO_NUM_ATTRS;

    
    if (H5VL_object_get(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, (-1), "unable to get attribute count for object");

    H5_CHECKED_ASSIGN(ret_value, int, oinfo.num_attrs, hsize_t);

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Aiterate1(hid_t loc_id, unsigned *attr_num , H5A_operator1_t op, void *op_data)
{
    H5VL_object_t                   *vol_obj = NULL; 
    H5VL_optional_args_t             vol_cb_args;    
    H5VL_native_attr_optional_args_t attr_opt_args;  
    herr_t                           ret_value;      

    FUNC_ENTER_API(H5_ITER_ERROR)

    
    if (H5I_ATTR == H5I_get_type(loc_id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5_ITER_ERROR, "location is not valid for an attribute");

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ATTR, H5E_BADTYPE, H5_ITER_ERROR, "invalid location identifier");

    
    attr_opt_args.iterate_old.loc_id   = loc_id;
    attr_opt_args.iterate_old.attr_num = attr_num;
    attr_opt_args.iterate_old.op       = op;
    attr_opt_args.iterate_old.op_data  = op_data;
    vol_cb_args.op_type                = H5VL_NATIVE_ATTR_ITERATE_OLD;
    vol_cb_args.args                   = &attr_opt_args;

    
    if ((ret_value = H5VL_attr_optional(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL)) <
        0)
        HERROR(H5E_ATTR, H5E_BADITER, "error iterating over attributes");

done:
    FUNC_LEAVE_API(ret_value)
} 
#endif 
