% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/read.r
\name{h5_read}
\alias{h5_read}
\title{Read an HDF5 Object or Attribute}
\usage{
h5_read(file, name = "/", attr = NULL, as = "auto")
}
\arguments{
\item{file}{The path to the HDF5 file.}

\item{name}{The full path of the dataset or group to read (e.g., \code{"/data/matrix"}).}

\item{attr}{The name of an attribute to read.
\itemize{
\item If \code{NULL} (default), the function reads the object specified by \code{name} (and attaches its attributes to the result).
\item If provided (string), the function reads \emph{only} the specified attribute from \code{name}.
}}

\item{as}{The target R data type.
\itemize{
\item \strong{Global:} \code{"auto"} (default), \code{"integer"}, \code{"double"}, \code{"logical"}, \code{"bit64"}, \code{"null"}.
\item \strong{Specific:} A named vector mapping names or type classes to R types (see Section "Type Conversion").
}}
}
\value{
An R object corresponding to the HDF5 object or attribute.
Returns \code{NULL} if the object is skipped via \code{as = "null"}.
}
\description{
Reads a dataset, a group, or a specific attribute from an HDF5 file into an R object.
}
\note{
The \code{@} prefix is \strong{only} used to configure attached attributes when reading a dataset (\code{attr = NULL}).
If you are reading a specific attribute directly (e.g., \code{h5_read(..., attr = "id")}), do \strong{not} use
the \code{@} prefix in the \code{as} argument.
}
\section{Type Conversion (\code{as})}{

You can control how HDF5 data is converted to R types using the \code{as} argument.

\strong{1. Mapping by Name:}
\itemize{
\item \code{as = c("data_col" = "integer")}: Reads the dataset/column named "data_col" as an integer.
\item \code{as = c("@validated" = "logical")}: When reading a dataset, this forces the attached attribute "validated" to be read as logical.
}

\strong{2. Mapping by HDF5 Type Class:}
You can target specific HDF5 data types using keys prefixed with a dot (\code{.}).
Supported classes include:
\itemize{
\item \strong{Integer:} \code{.int}, \code{.int8}, \code{.int16}, \code{.int32}, \code{.int64}
\item \strong{Unsigned:} \code{.uint}, \code{.uint8}, \code{.uint16}, \code{.uint32}, \code{.uint64}
\item \strong{Floating Point:} \code{.float}, \code{.float16}, \code{.float32}, \code{.float64}
}
Example: \code{as = c(.uint8 = "logical", .int = "bit64")}

\strong{3. Precedence & Attribute Config:}
\itemize{
\item \strong{Attributes vs Datasets:} Attribute type mappings take precedence over dataset mappings.
If you specify \code{as = c(.uint = "logical", "@.uint" = "integer")}, unsigned integer datasets
will be read as \code{logical}, but unsigned integer \emph{attributes} will be read as \code{integer}.
\item \strong{Specific vs Generic:} Specific keys (e.g., \code{.uint32}) take precedence over generic keys (e.g., \code{.uint}),
which take precedence over the global default (\code{.}).
}
}

\examples{
file <- tempfile(fileext = ".h5")

# --- Write Data ---
h5_write(c(10L, 20L), file, "ints")
h5_write(I(TRUE),     file, "ints", attr = "ready")
h5_write(c(10.5, 18), file, "floats")
h5_write(I("meters"), file, "floats", attr = "unit")

# --- Read Data ---
# Read dataset
x <- h5_read(file, "ints")
print(x)

# Read dataset with attributes
y <- h5_read(file, "floats")
print(attr(y, "unit"))

# Read a specific attribute directly
unit <- h5_read(file, "floats", attr = "unit")
print(unit)

# --- Type Conversion Examples ---

# Force integer dataset to be read as numeric (double)
x_dbl <- h5_read(file, "ints", as = "double")
class(x_dbl)

# Force attached attribute to be read as logical
# Note the "@" prefix to target the attribute
z <- h5_read(file, "ints", as = c("@ready" = "logical"))
print(z)

unlink(file)
}
\seealso{
\code{\link[=h5_write]{h5_write()}}
}
