#' @title Function to calculate the two-sample time-to-first-event (TTFE) statistics and variances across multiple calendar times.
#' @description
#' Computes sequential Z-statistics and their corresponding variance estimates for a
#' two-sample time-to-first-event (TTFE) analysis at a set of prespecified calendar
#' analysis times. At each calendar time, administrative censoring is applied, event
#' times are converted from the calendar scale to the event-time scale, and a standard
#' log-rank test is performed using only the first observed event per subject.
#'
#' @param data A data frame containing two-sample composite endpoint data, typically generated by \code{TwoSample.generate.sequential()}.
#' @param tau Optional upper bound for the event-time horizon. This argument is
#'   currently not used and is included for interface consistency with other
#'   sequential estimators in the package.
#' @param calendars A numeric vector of calendar times at which interim analyses are
#'   conducted. Each value is treated as an administrative censoring time.
#'
#' @returns A list with components:
#' \itemize{
#' \item \code{Z.stats}: A numeric vector of log-rank Z-statistics evaluated at each
#'   calendar analysis time.
#' \item \code{vars}: A numeric vector of estimated variances of the Z-statistics at
#'   each analysis time.
#'   \item \code{total.ns}: A numeric vector giving the total number of subjects
#'   contributing data at each calendar analysis time.
#' }
#' @export
#' @importFrom dplyr group_by filter mutate slice
#' @importFrom survival Surv survdiff
#'
TwoSample.Z.Var.Estimator.Sequential.TTFE <- function(data, tau = NULL, calendars){
  original.data <- data
  # output from this function:
  # 1. Z statistics at calendar times
  # 2. Estimated variances
  # 3. correlation matrix
  # 4. sample size at given calendar time
  Z.stats <- c()
  vars <- c()
  total.ns <- c()

  for (j in 1:length(calendars)){
    # j= 1
    # Step 1: Apply the calendar time as effective censoring time
    data.censored <- Apply.calendar.censoring.2(data = original.data, calendar = calendars[j])

    # Step 2: Run the estimator of the censored data
    # Keep the patients who are already in the study and convert the
    # event times from calendar scale to the event scale
    # For the TTFE approach, only keep the first event within each patient

    # keep the first event within each id
    data.censored <- data.censored %>%
      dplyr::group_by(.data$id) %>%
      dplyr::filter(!is.na(.data$status)) %>%
      dplyr::mutate(true_event_time = .data$event_time_cal - .data$e) %>%
      slice(1)

    fit <- survdiff(Surv(true_event_time, event) ~ group, data = data.censored)

    Z.stats[j] <- sqrt(fit$chisq)
    vars[j] <- fit$var[1,1]

    total.ns[j] <- length(unique(data.censored$id))


  } # end of the 'j' loop
  return(list(Z.stats = Z.stats,
              vars = vars,
              total.ns = total.ns))
}
