#' Test Reliability and CSEMs for IRT Scores
#'
#' @description
#' Compute test reliability for raw scores (and optionally scale scores),
#' along with associated conditional standard errors of measurement (CSEMs),
#' for a unidimensional IRT model.
#'
#' @param ip A data frame or matrix of item parameters. Columns are interpreted
#'   in order as:
#'   \itemize{
#'     \item 3 columns: \code{b}, \code{a}, \code{c} (3PL; \code{a} on the \code{D} metric),
#'     \item 2 columns: \code{b}, \code{a} (2PL; \code{c} internally set to 0),
#'     \item 1 column: \code{b} (1PL/Rasch; \code{a = 1}, \code{c = 0}).
#'   }
#' @param ct Optional. A data frame or matrix containing the score conversion
#'   table. If supplied, it must have \code{ni + 1} rows (for raw scores
#'   \code{0:ni}) and a column named \code{ss} giving the corresponding
#'   scale scores. If \code{ct = NULL} (default), only raw-score reliability and
#'   CSEMs are computed.
#' @param nq Integer. Number of quadrature points used to approximate the
#'   standard normal ability distribution. Defaults to \code{11}.
#' @param D Numeric. Scaling constant for the logistic IRT model. Defaults to
#'   \code{1.702}.
#'
#' @return A list with three components:
#' \describe{
#'   \item{fx}{A data frame containing the estimated marginal
#'     score distribution for raw scores (and scale scores if \code{ct} is
#'     provided).}
#'   \item{rel}{A data frame with overall error variance,
#'     true score variance, observed score variance, and reliability for raw
#'     scores, and additionally for scale scores if \code{ct} is provided.}
#'   \item{csem}{A data frame with theta, weights, expected raw
#'     scores and corresponding CSEMs. If \code{ct} is provided, expected scale
#'     scores and scale-score CSEMs are also included.}
#' }
#' @examples
#' data(ip.u)
#' data(ct.u)
#' rel_test(ip.u)
#' rel_test(ip.u, ct.u)
#'
#' @export
rel_test <- function(ip, ct = NULL, nq = 11, D = 1.702) {

  ## ---- check ip -------------------------------------------------------------
  if (missing(ip)) {
    stop("`ip` must be supplied as a data frame or matrix of item parameters.")
  }
  if (!is.data.frame(ip) && !is.matrix(ip)) {
    stop("`ip` must be a data frame or a matrix.")
  }

  ip <- as.data.frame(ip)
  if (!all(vapply(ip, is.numeric, logical(1L)))) {
    stop("All columns in `ip` must be numeric.")
  }
  if (ncol(ip) < 1L || ncol(ip) > 3L) {
    stop("`ip` must have 1, 2, or 3 columns (b[, a[, c]]).")
  }

  # standardize to b, a, c
  if (ncol(ip) == 3L) {
    names(ip) <- c("b", "a", "c")
  } else if (ncol(ip) == 2L) {
    names(ip) <- c("b", "a")
    ip$c <- 0
  } else { # 1 column
    names(ip) <- "b"
    ip$a <- 1
    ip$c <- 0
  }

  ni <- nrow(ip)
  if (ni < 1L) {
    stop("`ip` must contain at least one item (one row).")
  }

  ## ---- check ct if provided -------------------------------------------------
  has_scale <- !is.null(ct)

  if (has_scale) {
    if (!is.data.frame(ct) && !is.matrix(ct)) {
      stop("`ct` must be a data frame or a matrix if provided.")
    }
    ct <- as.data.frame(ct)

    if (!("ss" %in% names(ct))) {
      stop("`ct` must contain a column named `ss` for scale scores.")
    }
    if (nrow(ct) != (ni + 1L)) {
      stop("`ct` must have `ni + 1` rows corresponding to raw scores 0:ni.")
    }
  }

  ## ---- quadrature points and weights ----------------------------------------
  # assumes normal_quadra() returns list(nodes, weights) for standard normal
  quad <- normal_quadra(n = nq, mm = 4)
  theta_nodes <- quad$nodes
  weights     <- quad$weights

  ## ---- compute item response probabilities P(theta) -------------------------
  b <- ip$b
  a <- ip$a
  c <- ip$c

  # theta matrix (nq x ni), each row = one theta, each column = one item
  theta_mat <- matrix(theta_nodes, nrow = nq, ncol = ni)
  b_mat     <- matrix(rep(b, each = nq), nrow = nq, ncol = ni)
  a_mat     <- matrix(rep(a, each = nq), nrow = nq, ncol = ni)
  c_mat     <- matrix(rep(c, each = nq), nrow = nq, ncol = ni)

  P_mat <- c_mat + (1 - c_mat) / (1 + exp(-D * a_mat * (theta_mat - b_mat)))

  ## ---- conditional raw-score distribution P(X = x | theta) ------------------
  # fxTheta: rows = theta nodes, columns = raw scores 0..ni
  fxTheta <- matrix(NA_real_, nrow = nq, ncol = ni + 1L)
  for (i in seq_len(nq)) {
    probs_i <- P_mat[i, ]
    fxTheta[i, ] <- lord_wingersky(probs_i)$probability
  }

  raw_scores <- 0:ni
  fxTheta_df <- as.data.frame(fxTheta)
  names(fxTheta_df) <- paste0("X", raw_scores)
  fxTheta_df$theta   <- theta_nodes
  fxTheta_df$weights <- weights

  ## ---- expected scores at each theta ----------------------------------------
  # E[X | theta]
  Ex_raw <- as.numeric(fxTheta %*% raw_scores)

  if (has_scale) {
    # E[Scale | theta]
    Ex_scale <- as.numeric(fxTheta %*% ct$ss)
  }

  fxTheta_df$Ex_raw <- Ex_raw
  if (has_scale) {
    fxTheta_df$Ex_scale <- Ex_scale
  }

  ## ---- true-score variances -------------------------------------------------
  mean_raw <- sum(Ex_raw * weights)
  var_true_raw <- sum(weights * (Ex_raw - mean_raw)^2)

  if (has_scale) {
    mean_scale <- sum(Ex_scale * weights)
    var_true_scale <- sum(weights * (Ex_scale - mean_scale)^2)
  }

  ## ---- conditional error variances at each theta ----------------------------
  var_raw <- numeric(nq)
  if (has_scale) {
    var_scale <- numeric(nq)
  }

  for (i in seq_len(nq)) {
    p_x_given_theta <- fxTheta[i, ]
    var_raw[i] <- sum(p_x_given_theta * (raw_scores - Ex_raw[i])^2)

    if (has_scale) {
      var_scale[i] <- sum(p_x_given_theta * (ct$ss - Ex_scale[i])^2)
    }
  }

  fxTheta_df$Var_raw <- var_raw
  if (has_scale) {
    fxTheta_df$Var_scale <- var_scale
  }

  # overall error variances
  var_err_raw <- sum(var_raw * weights)
  var_obs_raw <- var_true_raw + var_err_raw

  if (has_scale) {
    var_err_scale <- sum(var_scale * weights)
    var_obs_scale <- var_true_scale + var_err_scale
  }

  ## ---- marginal score distribution (raw, and scale if available) ------------
  # P(X = x) = integral P(X = x | theta) f(theta) dtheta ≈ sum_i p(x|theta_i)*w_i
  fxTheta_weighted <- fxTheta * weights
  w_raw <- colSums(fxTheta_weighted)  # length ni+1, sums to 1

  fitted_freq <- data.frame(
    wts = w_raw,
    x   = raw_scores
  )

  if (has_scale) {
    fitted_freq$ss <- ct$ss
  }

  ## ---- reliability ----------------------------------------------------------
  rel_raw <- 1 - var_err_raw / var_obs_raw

  if (has_scale) {
    rel_scale <- 1 - var_err_scale / var_obs_scale
  }

  ## ---- variance and reliability summary -------------------------------------
  vals <- c(
    var_err_raw,
    var_true_raw,
    var_obs_raw,
    rel_raw
  )
  row_names <- c(
    "Overall error variance for raw scores",
    "True score variance for raw scores",
    "Observed score variance for raw scores",
    "Reliability for raw scores"
  )

  if (has_scale) {
    vals <- c(
      vals,
      var_err_scale,
      var_true_scale,
      var_obs_scale,
      rel_scale
    )
    row_names <- c(
      row_names,
      "Overall error variance for scale scores",
      "True score variance for scale scores",
      "Observed score variance for scale scores",
      "Reliability for scale scores"
    )
  }

  VarandRel <- data.frame(coefficient = vals)
  rownames(VarandRel) <- row_names

  ## ---- conditional SEMs output ----------------------------------------------
  if (has_scale) {
    conditionalSEMs <- data.frame(
      theta       = fxTheta_df$theta,
      wts     = fxTheta_df$weights,
      ex_raw      = fxTheta_df$Ex_raw,
      ex_scale    = fxTheta_df$Ex_scale,
      raw_csem    = sqrt(fxTheta_df$Var_raw),
      scale_csem  = sqrt(fxTheta_df$Var_scale)
    )
  } else {
    conditionalSEMs <- data.frame(
      theta    = fxTheta_df$theta,
      wts  = fxTheta_df$weights,
      ex_raw   = fxTheta_df$Ex_raw,
      raw_csem = sqrt(fxTheta_df$Var_raw)
    )
  }
  rownames(conditionalSEMs) <- seq_len(nrow(conditionalSEMs))

  ## ---- return ---------------------------------------------------------------
  return(list(
    "fx"       = fitted_freq,
    "rel" = VarandRel,
    "csem"         = conditionalSEMs
  ))
}
