% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dtms_expectancy.R
\name{dtms_expectancy}
\alias{dtms_expectancy}
\title{Calculate state expectancy}
\usage{
dtms_expectancy(
  probs = NULL,
  matrix = NULL,
  dtms,
  risk = NULL,
  start_distr = NULL,
  start_time = NULL,
  start_state = NULL,
  end_time = NULL,
  correction = 0.5,
  total = TRUE,
  fundamental = FALSE,
  verbose = FALSE
)
}
\arguments{
\item{probs}{Data frame with transition probabilities, as created with \code{dtms_transitions}.}

\item{matrix}{Matrix with transition probabilities, as generated with \code{dtms_matrix}.}

\item{dtms}{dtms object, as created with \code{dtms}.}

\item{risk}{Character (otpional), name of one transient state. If specified expectancies are only shown for this state but by values of the time scale.}

\item{start_distr}{Numeric (optional), distribution of starting states. If specified, average expectancy over all starting states will be calculated. Only applied if risk=NULL.}

\item{start_time}{Numeric (optional), value of time scale for start. If NULL (default) first value of time scale will be used.}

\item{start_state}{Character (optional), name of starting states. If NULL (default) all transient states will be used.}

\item{end_time}{Numeric (optional), last value of time scale to consider. If NULL (default) all values of time scale starting from start_time will be used.}

\item{correction}{Numeric (optional), correction for expectancy when starting state and state under consideration match, see details. Defaults to 0.5.}

\item{total}{Logical (optional), calculate total expectancy. Default is TRUE. Only applied if risk=NULL.}

\item{fundamental}{Logical (optional), return fundamental matrix? Default is FALSE.}

\item{verbose}{Logical (optional), print some information on what is computed. Default is FALSE.}
}
\value{
A matrix with state expectancies.
}
\description{
This function calculates the expected time spent in the transient states
(state expectancy).
}
\details{
If the argument `start_distr` is specified, the average of the state
expectancies over all starting states is calculated. The names and length
of `start_distr` need to match the starting states generated by this
function which are based on the `dtms` object.

The partial expectancy for the time spent in the transient states can be
calculated using the arguments `start_time` and `end_time`.

IF the argument `risk` is specified, then only the remaining life expectancy
for the state specified with this argument is shown, but for all time units
of the time scale.

Two corrections to the results will be applied per default. Both corrections
are required as the underlying formulas do actually not provide the
expected time spent in a state, but the number of visits to a state. Time
and visits are only equal under certain conditions; in particular, only if
transitions between states happen mid-interval and the step length of the
time scale is equal to one. The first correction will remove a certain amount
of time spent in a certain state if its equal to the starting state. This is
controlled with the argument `correction` which is applied multiplicative.
For instance, its default value 0.5 means that the state expectancy in some
state X starting from state X is reduced by 0.5 time steps. The second
correction uses the entry `timestep` of `dtms`, and multiplies results with its value.
}
\examples{
## Define model: Absorbing and transient states, time scale
simple <- dtms(transient=c("A","B"),
               absorbing="X",
               timescale=0:20)
## Reshape to transition format
estdata <- dtms_format(data=simpledata,
                       dtms=simple,
                       idvar="id",
                       timevar="time",
                       statevar="state")
## Clean
estdata <- dtms_clean(data=estdata,
                      dtms=simple)
# Fit model
fit <- dtms_fit(data=estdata)
## Predict probabilities
probs    <- dtms_transitions(dtms=simple,
                             model = fit)
## Get starting distribution
S <- dtms_start(dtms=simple,
                data=estdata)
## State expectancies
dtms_expectancy(dtms=simple,
                probs=probs,
                start_distr=S)
}
