% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compute_mean.R
\name{compute_mean}
\alias{compute_mean}
\title{Calculate mean in temporal or spatial domain}
\usage{
compute_mean(
  data,
  amplitude = "signal_base",
  domain = c("time", "space"),
  level = c("epoch", "condition", "sensor", "subject", "group"),
  type = c("point", "jack"),
  weights_col = NULL,
  R = NULL,
  alpha = 0.95
)
}
\arguments{
\item{data}{A data frame, tibble or a database table with input data, required columns: \code{time} or \code{sensor} (according to the selected domain), the column with the EEG amplitude specified in the argument \code{amplitude} and columns corresponding to the selected \code{level}.}

\item{amplitude}{A character specifying the name of the column from input data with an EEG amplitude values. Default is \code{"signal_base"} for computing average from baseline corrected signal.}

\item{domain}{A character specifying the domain over which the average is computed. One of \code{"time"} or \code{"space"}. Option \code{"time"} computes a time-resolved average at each time point, whereas \code{"space"} computes a space-resolved average at each sensor.}

\item{level}{A character specifying the level of average calculation. The possible values are \code{"epoch"},\code{"condition"}, \code{"sensor"}, \code{"subject"} and \code{"group"}. See Details for more information.}

\item{type}{A character specifying the method of calculating the average, \code{"point"} for pointwise average and \code{"jack"} for jackknife leave-one-out average.}

\item{weights_col}{A character specifying the name of the column containing observation weights. If \code{NULL}, un-weighted standard pointwise average is computed.}

\item{R}{The number of replications used in bootstrap interval calculation. Required only for computing pointwise mean. Default value is 1000.}

\item{alpha}{A number indicating confidence interval level. The default value is 0.95 for 95\% confidence intervals.}
}
\value{
A tibble with resulting average and CI bounds according to the chosen \code{level}, \code{domain} and \code{alpha} arguments. The statistics are saved in columns
\itemize{
\item \code{average} for computed average amplitude value,
\item \code{n} for number of observations used in average computing,
\item \code{se} for standard error of the mean,
\item \code{ci_low} for lower bound of the confidence interval and
\item \code{ci_up} for upper bound of the confidence interval.
}
}
\description{
Calculate a pointwise or a jackknife (leave-one-out) average signal across temporal or spatial domain together with standard error and pointwise confidence interval (CI) bounds.
Pointwise averages can be computed in two ways: standard (un-weighted) by default, or weighted using the values in the column specified by \code{weights_col}.

The function computes an average at group, subject, sensor/time point, condition or epoch level (according to the \code{level} parameter). For the option \code{level = "epoch"} the epochs are averaged etc. Function assumes pre-prepared data according to the chosen level.
}
\details{
The function supports averaging at different hierarchical levels of the data (using \code{level} argument):
\itemize{
\item \code{"epoch"}: averaging across epochs. Returns the average curve (time domain) or sensor array (space domain) for each combination of other grouping variables.
\item \code{"condition"}: averages across experimental conditions.
\item \code{"sensor"}: averages across sensors (space domain) or time points (time domain).
\item \code{"subject"}: averages across subjects.
\item \code{"group"}: averages across groups of subjects (highest aggregation level).
The function assumes input adapted to the desired level of averaging (i.e. for \code{epoch} level the \code{epoch} column must be present in \code{data} etc.). For all levels higher than epochs, the averages of the lower level are assumed as the input data.
}

Weighted vs un-weighted average (\code{type = "point"}):
\itemize{
\item If \code{weights_col} is \code{NULL}, each observation is treated equally (with weight = 1), producing a standard un-weighted mean, standard errors (SE), and CI.
\item If \code{weight_cols} is provided, a weighted average is computed using the values in the specified column as weights. SE and CI are computed based on the weighted variance.
}

Computing standard error of the mean:
\itemize{
\item For \code{type = "point"}, the standard error is computed as sample standard deviation divided by square root of the sample size for standard mean or its weighted alternative (if \code{weights_col} is specified).
\item For \code{type = "jack"}, the standard error is jackknife standard error of the mean (for the exact formula see Efron and Tibshirani 1994).
}

Computing point confidence intervals:
For each average value, the upper and lower bounds of the point confidence interval are also available.
\itemize{
\item Setting \code{type = "point"} and \code{R}: the bounds are computed using percentile method from bootstrapping with \code{R} replicates (can be slow for large amounts of data).
\item Setting \code{type = "point"} without specifying \code{R}: the bounds are computed using standard error of the mean and approximation by the Student distribution.
\item Setting \code{type = "jack"}: the bounds are computed using jackknife standard error of the mean and approximation by the Student t-distribution. Note: used method assumes equal variance and symmetric distribution, which may be problematic for very small samples.
}

Note: If there are \code{NA}'s in \code{amplitude} or \code{weights_col} columns, corresponding rows are ignored in the average calculation and function prints a warning message.
}
\examples{
# Average (pointwise) raw signal for subject 1 and electrode E1
# without outlier epoch 14
avg_data <- epochdata |>
pick_data(subject_rg = 1, epoch_rg = 1:13, sensor_rg = "E1") |>
compute_mean(amplitude = "signal", level = "epoch", domain = "time")
str(avg_data)
\donttest{
# plot the result using interactive plot with pointwise CI
avg_data |>
pick_data(subject = 1) |>
interactive_waveforms(amplitude = "average", t0 = 10,
level = "sensor", avg = FALSE, CI = TRUE)
}

# Jackknife average signal for subject 1 in all electrodes in time point 11 with baseline correction
# on interval 1:10 (again without outlier epoch 14)
# a) prepare corrected data
basedata <- pick_data(epochdata, subject_rg = 1) |>
baseline_correction(baseline_range = 1:10, type = "absolute")
# b) filter time point 11 (without epoch 14) and compute the average
avg_data <- pick_data(basedata, time_rg = 11, epoch_rg = 1:13) |>
compute_mean(amplitude = "signal_base", level = "epoch", domain = "space", type = "jack")
str(avg_data)
# c) plot the result with topo_plot()
topo_plot(data = avg_data, amplitude = "average")

# Space average on subject level (average for all included subjects in time point 11)
# a) compute mean from all epochs for each subject
mean_epoch <- epochdata |>
pick_data(time_rg = 11, epoch_rg = 1:13) |>
compute_mean(amplitude = "signal", level = "epoch", domain = "space", type = "point")
# b) compute mean on subject level
mean_subjects <- compute_mean(mean_epoch, amplitude = "average", level = "subject",
domain = "space", type = "point")
head(mean_subjects)
# c) compute weighted mean with number of observations as weights
weighted_mean_subjects <- compute_mean(mean_epoch, amplitude = "average", level = "subject",
domain = "space", type = "point", weights_col = "n")
head(weighted_mean_subjects)
}
\references{
Efron B., Tibshirani RJ. \emph{An Introduction to the Bootstrap.} Chapman & Hall/CRC; 1994.
}
