% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_rates_vs_trait_data_for_focal_time.R
\name{plot_rates_vs_trait_data_for_focal_time}
\alias{plot_rates_vs_trait_data_for_focal_time}
\title{Plot rates vs. trait data for a given focal time}
\usage{
plot_rates_vs_trait_data_for_focal_time(
  deepSTRAPP_outputs,
  focal_time = NULL,
  rate_type = "net_diversification",
  select_trait_levels = "all",
  color_scale = NULL,
  colors_per_levels = NULL,
  display_plot = TRUE,
  PDF_file_path = NULL,
  return_mean_rates_vs_trait_data_df = FALSE
)
}
\arguments{
\item{deepSTRAPP_outputs}{List of elements generated with \code{\link[=run_deepSTRAPP_for_focal_time]{run_deepSTRAPP_for_focal_time()}},
that summarize the results of a STRAPP test for a specific time in the past (i.e. the \code{focal_time}).
\code{deepSTRAPP_outputs} can also be extracted from the output of \code{\link[=run_deepSTRAPP_over_time]{run_deepSTRAPP_over_time()}} that
runs the whole deepSTRAPP workflow over multiple time-steps.}

\item{focal_time}{Numerical. (Optional) If \code{deepSTRAPP_outputs} comprises results over multiple time-steps
(i.e., output of \code{\link[=run_deepSTRAPP_over_time]{run_deepSTRAPP_over_time()}}, this is the time of the STRAPP test targeted for plotting.}

\item{rate_type}{A character string specifying the type of diversification rates to plot.
Must be one of 'speciation', 'extinction' or 'net_diversification' (default).
Even if the \code{deepSTRAPP_outputs} object was generated with \code{\link[=run_deepSTRAPP_over_time]{run_deepSTRAPP_over_time()}}
for testing another type of rates, the object will contain data for all types of rates.}

\item{select_trait_levels}{(Vector of) character string. Only for categorical and biogeographic trait data.
To provide a list of a subset of states/ranges to plot. Names must match the ones found in the \code{deepSTRAPP_outputs}.
Default is \code{all} which means all states/ranges will be plotted.}

\item{color_scale}{Vector of character string. List of colors to use to build the color scale with \code{\link[grDevices:colorRamp]{grDevices::colorRampPalette()}}
to display the points. Color scale from lowest values to highest rate values. Only for continuous data.
Default = \code{NULL} will use the 'Spectral' color palette in \code{\link[RColorBrewer:ColorBrewer]{RColorBrewer::brewer.pal()}}.}

\item{colors_per_levels}{Named character string. To set the colors to use to plot data points and box for each state/range. Names = states/ranges; values = colors.
If \code{NULL} (default), the default ggplot2 color palette (\code{\link[scales:pal_hue]{scales::hue_pal()}}) will be used. Only for categorical and biogeographic data.}

\item{display_plot}{Logical. Whether to display the plot generated in the R console. Default is \code{TRUE}.}

\item{PDF_file_path}{Character string. If provided, the plot will be saved in a PDF file following the path provided here. The path must end with ".pdf".}

\item{return_mean_rates_vs_trait_data_df}{Logical. Whether to include in the output the data.frame of mean rates per trait values/states/ranges computed for
each posterior sample at the focal time. Default is \code{FALSE}.}
}
\value{
The function returns a list with at least one element.
\itemize{
\item \code{rates_vs_trait_ggplot} An object of classes \code{gg} and \code{ggplot}. This is a ggplot that can be displayed
on the console with \code{print(output$rates_vs_trait_ggplot)}. It corresponds to the plot being displayed on the console
when the function is run, if \code{display_plot = TRUE}, and can be further modify for aesthetics using the ggplot2 grammar.
}

If the trait data are 'continuous', the plot is a scatter plot showing how diversification rates varies with trait values.
If the trait data are 'categorical' or 'biogeographic', the plot is a boxplot showing diversification rates per states/ranges.

Each plot also displays summary statistics for the STRAPP test associated with the data displayed:
\itemize{
\item An observed statistic computed across the mean traits/ranges and rates values shown on the plot. This is not the statistic of the STRAPP test itself,
which is conducted across all BAMM posterior samples.
\item The quantile of null statistic distribution at the significant threshold used to define test significance. The test will be considered significant
(i.e., the null hypothesis is rejected) if this value is higher than zero.
\item The p-value of the associated STRAPP test.
}

Optional summary data.frame:
\itemize{
\item \code{mean_rates_vs_trait_data_df} A data.frame with three columns providing the \verb{$mean_rates} and \verb{$trait_value}
observed along branches at \code{focal_time}. Rates are averaged across all BAMM posterior samples.
This is the raw data used to draw the plot. Included if \code{return_mean_rates_vs_trait_data_df = TRUE}.
}

If a \code{PDF_file_path} is provided, the function will also generate a PDF file of the plot.
}
\description{
Plot rates vs. trait data as extracted for a given focal time.
Data are extracted from the output of a deepSTRAPP run carried out with
\code{\link[=run_deepSTRAPP_for_focal_time]{run_deepSTRAPP_for_focal_time()}} or
\code{\link[=run_deepSTRAPP_over_time]{run_deepSTRAPP_over_time()}}).

Returns a single plot showing rates vs. trait data for a given focal time.
If the trait data are 'continuous', the plot is a scatter plot.
If the trait data are 'categorical' or 'biogeographic', the plot is a boxplot.

If a PDF file path is provided in \code{PDF_file_path}, the plot will be saved directly in a PDF file.
}
\details{
The main input \code{deepSTRAPP_outputs} is the typical output of \code{\link[=run_deepSTRAPP_for_focal_time]{run_deepSTRAPP_for_focal_time()}}.
It provides information on results of a STRAPP test performed at a given \code{focal_time}.

Plots are built based on both trait data and diversification data as extracted for the given \code{focal_time}.
Such data are recorded in the outputs of a deepSTRAPP run carried out with \code{\link[=run_deepSTRAPP_for_focal_time]{run_deepSTRAPP_for_focal_time()}}
when \code{return_updated_trait_data_with_Map = TRUE} for trait data, and \code{extract_diversification_data_melted_df = TRUE} for diversification data.
Please ensure to select those arguments when running deepSTRAPP.

Alternatively, the main input \code{deepSTRAPP_outputs} can be the output of \code{\link[=run_deepSTRAPP_over_time]{run_deepSTRAPP_over_time()}},
providing results of STRAPP tests over multiple time-steps. In this case, you must provide a \code{focal_time} to select the
unique time-step used for plotting.
\itemize{
\item \code{return_updated_trait_data_with_Map} must be set to \code{TRUE} so that the trait data used to compute the tests are returned among the outputs
under \verb{$updated_trait_data_with_Map_over_time}. Alternatively, and more efficiently, \code{extract_trait_data_melted_df} can be set to \code{TRUE}
so that trait data are already returned in a melted data.frame among the outputs under \verb{$trait_data_df_over_time}.
\item \code{extract_diversification_data_melted_df} must be set to \code{TRUE} so that the diversification rates are returned
among the outputs under \verb{$diversification_data_df_over_time}.
}

For plotting all time-steps at once, see \code{\link[=plot_rates_vs_trait_data_over_time]{plot_rates_vs_trait_data_over_time()}}.
}
\examples{
if (deepSTRAPP::is_dev_version())
{
 # ----- Example 1: Continuous trait ----- #

 # Load fake trait df
 data(Ponerinae_trait_tip_data, package = "deepSTRAPP")
 # Load phylogeny with old calibration
 data(Ponerinae_tree_old_calib, package = "deepSTRAPP")

 # Load the BAMM_object summarizing 1000 posterior samples of BAMM
 data(Ponerinae_BAMM_object_old_calib, package = "deepSTRAPP")
 ## This dataset is only available in development versions installed from GitHub.
 # It is not available in CRAN versions.
 # Use remotes::install_github(repo = "MaelDore/deepSTRAPP") to get the latest development version.

 ## Prepare trait data

 # Extract continuous trait data as a named vector
 Ponerinae_cont_tip_data <- setNames(object = Ponerinae_trait_tip_data$fake_cont_tip_data,
                                     nm = Ponerinae_trait_tip_data$Taxa)

 # Select a color scheme from lowest to highest values
 color_scale = c("darkgreen", "limegreen", "orange", "red")

 # Get Ancestral Character Estimates based on a Brownian Motion model
 # To obtain values at internal nodes
 Ponerinae_ACE <- phytools::fastAnc(tree = Ponerinae_tree_old_calib, x = Ponerinae_cont_tip_data)

 \donttest{ # (May take several minutes to run)
 # Run a Stochastic Mapping based on a Brownian Motion model
 # to interpolate values along branches and obtain a "contMap" object
 Ponerinae_contMap <- phytools::contMap(Ponerinae_tree_old_calib, x = Ponerinae_cont_tip_data,
                                        res = 100, # Number of time steps
                                        plot = FALSE)
 # Plot contMap = stochastic mapping of continuous trait
 plot_contMap(contMap = Ponerinae_contMap,
              color_scale = color_scale)

 ## Set focal time to 10 Mya
 focal_time <- 10

 ## Run deepSTRAPP on net diversification rates for focal time = 10 Mya.

 Ponerinae_deepSTRAPP_cont_old_calib_10My <- run_deepSTRAPP_for_focal_time(
    contMap = Ponerinae_contMap,
    ace = Ponerinae_ACE,
    tip_data = Ponerinae_cont_tip_data,
    trait_data_type = "continuous",
    BAMM_object = Ponerinae_BAMM_object_old_calib,
    focal_time = focal_time,
    rate_type = "net_diversification",
    return_perm_data = TRUE,
    # Need to be set to TRUE to save diversification data
    extract_diversification_data_melted_df = TRUE,
    # Need to be set to TRUE to save trait data
    return_updated_trait_data_with_Map = TRUE,
    return_updated_BAMM_object = TRUE)

 ## Explore output
 str(Ponerinae_deepSTRAPP_cont_old_calib_10My, max.level = 1)

 # ----- Plot histogram of STRAPP overall test results from run_deepSTRAPP_for_focal_time() ----- #

 # Get plot
 rates_vs_trait_output <- plot_rates_vs_trait_data_for_focal_time(
    deepSTRAPP_outputs = deepPonerinae_deepSTRAPP_cont_old_calib_10My,
    color_scale = c("grey80", "orange"),
    display_plot = TRUE,
    # PDF_file_path = "./plot_rates_vs_trait_10My.pdf"
    return_mean_rates_vs_trait_data_df = TRUE)
 # Adjust aesthetics a posteriori
 rates_vs_trait_ggplot_adj <- rates_vs_trait_output$rates_vs_trait_ggplot +
    ggplot2::theme(plot.title = ggplot2::element_text(color = "red", size = 15))
 print(rates_vs_trait_ggplot_adj)

 # Explore melted data.frame of mean rates and trait values extracted for the given focal time.
 head(rates_vs_trait_output$mean_rates_vs_trait_data_df) }

 # ----- Plot histogram of STRAPP overall test results from run_deepSTRAPP_over_time() ----- #

 ## Load directly outputs from run_deepSTRAPP_over_time()
 data(Ponerinae_deepSTRAPP_cont_old_calib_0_40, package = "deepSTRAPP")
 ## This dataset is only available in development versions installed from GitHub.
 # It is not available in CRAN versions.
 # Use remotes::install_github(repo = "MaelDore/deepSTRAPP") to get the latest development version.

 # Select focal_time = 10My
 focal_time <- 10

 # Get plot
 rates_vs_trait_output <- plot_rates_vs_trait_data_for_focal_time(
    deepSTRAPP_outputs = Ponerinae_deepSTRAPP_cont_old_calib_0_40,
    focal_time = focal_time,
    color_scale = c("grey80", "purple"),
    display_plot = TRUE)
    # PDF_file_path = "./plot_rates_vs_trait_10My.pdf"

 # Adjust aesthetics a posteriori
 rates_vs_trait_ggplot_adj <- rates_vs_trait_output$rates_vs_trait_ggplot +
     ggplot2::theme(plot.title = ggplot2::element_text(color = "red", size = 15))
 print(rates_vs_trait_ggplot_adj)


 # ----- Example 2: Categorical trait ----- #

 ## Load data

 # Load phylogeny
 data(Ponerinae_tree, package = "deepSTRAPP")
 # Load trait df
 data(Ponerinae_trait_tip_data, package = "deepSTRAPP")

 # Load the BAMM_object summarizing 1000 posterior samples of BAMM
 data(Ponerinae_BAMM_object_old_calib, package = "deepSTRAPP")
 ## This dataset is only available in development versions installed from GitHub.
 # It is not available in CRAN versions.
 # Use remotes::install_github(repo = "MaelDore/deepSTRAPP") to get the latest development version.

 ## Prepare trait data

 # Extract categorical data with 3-levels
 Ponerinae_cat_3lvl_tip_data <- setNames(object = Ponerinae_trait_tip_data$fake_cat_3lvl_tip_data,
                                         nm = Ponerinae_trait_tip_data$Taxa)
 table(Ponerinae_cat_3lvl_tip_data)

 # Select color scheme for states
 colors_per_states <- c("forestgreen", "sienna", "goldenrod")
 names(colors_per_states) <- c("arboreal", "subterranean", "terricolous")

 \donttest{ # (May take several minutes to run)
 ## Produce densityMaps using stochastic character mapping based on an ARD Mk model
 Ponerinae_cat_3lvl_data_old_calib <- prepare_trait_data(
    tip_data = Ponerinae_cat_3lvl_tip_data,
    phylo = Ponerinae_tree_old_calib,
    trait_data_type = "categorical",
    colors_per_states = colors_per_states,
    evolutionary_models = "ARD", # Use default ARD model
    nb_simulations = 100, # Reduce number of simulations to save time
    seed = 1234, # Seet seed for reproducibility
    return_best_model_fit = TRUE,
    return_model_selection_df = TRUE,
    plot_map = FALSE)

 # Load directly output
 data(Ponerinae_cat_3lvl_data_old_calib, package = "deepSTRAPP")

 ## Set focal time to 10 Mya
 focal_time <- 10

 ## Run deepSTRAPP on net diversification rates for focal time = 10 Mya.

 Ponerinae_deepSTRAPP_cat_3lvl_old_calib_10My <- run_deepSTRAPP_for_focal_time(
    densityMaps = Ponerinae_cat_3lvl_data_old_calib$densityMaps,
    ace = Ponerinae_cat_3lvl_data_old_calib$ace,
    tip_data = Ponerinae_cat_3lvl_tip_data,
    trait_data_type = "categorical",
    BAMM_object = Ponerinae_BAMM_object_old_calib,
    focal_time = focal_time,
    rate_type = "net_diversification",
    posthoc_pairwise_tests = TRUE,
    return_perm_data = TRUE,
    # Need to be set to TRUE to save diversification data
    extract_diversification_data_melted_df = TRUE,
    # Need to be set to TRUE to save trait data
    return_updated_trait_data_with_Map = TRUE,
    return_updated_BAMM_object = TRUE)

 ## Explore output
 str(Ponerinae_deepSTRAPP_cat_3lvl_old_calib_10My, max.level = 1)

 ## Plot rates vs. states
 rates_vs_trait_output <- plot_rates_vs_trait_data_for_focal_time(
    deepSTRAPP_outputs = Ponerinae_deepSTRAPP_cat_3lvl_old_calib_10My,
    focal_time = 10,
    select_trait_levels = c("arboreal", "terricolous"), # Select only two levels
    colors_per_levels = colors_per_states[c("arboreal", "terricolous")], # Adjust colors
    display_plot = TRUE,
    # PDF_file_path = "./plot_rates_vs_trait_10My.pdf",
    return_mean_rates_vs_trait_data_df = TRUE)

 # Adjust aesthetics a posteriori
 rates_vs_trait_ggplot_adj <- rates_vs_trait_output$rates_vs_trait_ggplot +
    ggplot2::theme(plot.title = ggplot2::element_text(color = "red", size = 15))
 print(rates_vs_trait_ggplot_adj)

 # Explore melted data.frame of mean rates and states extracted for the given focal time.
 head(rates_vs_trait_output$mean_rates_vs_trait_data_df) }
 }

}
\seealso{
Associated functions in deepSTRAPP: \code{\link[=run_deepSTRAPP_for_focal_time]{run_deepSTRAPP_for_focal_time()}} \code{\link[=plot_rates_vs_trait_data_over_time]{plot_rates_vs_trait_data_over_time()}}
}
\author{
Maël Doré
}
