% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ODEs.R
\name{BVPshoot2}
\alias{BVPshoot2}
\title{Solves a second-order BVP using the shooting method}
\usage{
BVPshoot2(f, t0, tf, y0, yf, h, s_guess = 1, tol = 1e-09, ...)
}
\arguments{
\item{f}{A function of the form \code{f(t,y,dy)} returning a numeric 
scalar. This defines the second-order ODE.}

\item{t0}{Initial time.}

\item{tf}{Final time.}

\item{y0}{Boundary value at \eqn{t_0}, i.e. \eqn{y(t_0)=y_0}.}

\item{yf}{Boundary value at \eqn{t_f}, i.e. \eqn{y(t_f)=y_f}.}

\item{h}{Step size for the RK4 integration.}

\item{s_guess}{A numeric starting guess for \eqn{y'(t_0)} (default is 1),
or a 2-element numeric vector giving a bracketing interval.}

\item{tol}{A numeric value that tells the bisection algorithms when to
stop. Default is 1e-9.}

\item{...}{Additional arguments passed to \code{f}.}
}
\value{
A list with elements \code{t} (time grid) and \code{y} (solution 
        matrix), where \code{y[,1]} contains \eqn{y(t)} and \code{y[,2]} 
        its derivative.
}
\description{
Solves \eqn{y^{('')} = f(t,y,y^{'})} on the interval \eqn{[t_0,t_f]} with
boundary conditions \eqn{y(t_0) = y_0}, \eqn{y(t_f) = y_f}, using the
shooting method and a root finder. The associated IVP is solved using
4th-order Runge-Kutta with \code{RK4ODE}. The second initial value is
found using the bisection method with \code{roots_bisec}.
}
\details{
It is important to consider the uniqueness of the solution of a BVP.
If the BVP admits infinitely many solutions (a family of solutions),
\code{BVPshoot2} will find only one of them, depending on what initial
condition for the first derivative of the associated IVP, was found
using the bisection method.
}
\examples{
        
# y"+ y = 9*sin(2*t); y(0)=-1, y(3*pi/2)=0
# Unique solution: y(t) = 3*sin(2*t) - cos(t)

# Define y"=f(t,y,y')
f <- function(t,y,dy) {ff <- -9*sin(2*t)-y; return(ff)}

# Solution interval
t0 <- 0
tf <- 3*pi/2

# Boundary values
y0 <- -1
yf <- 0

# Step size
h <- 0.01

# Solution
ltmp <- BVPshoot2(f,t0,tf,y0,yf,h)

# Check
# Number of steps
n <- length(ltmp$t)-1
print(c(ltmp$t[1],ltmp$t[n+1]))
print(c(ltmp$y[1,1],ltmp$y[n+1,1]))

}
