% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/class-network.R
\name{as_cograph}
\alias{as_cograph}
\title{Convert to Cograph Network}
\usage{
as_cograph(x, directed = NULL, ...)
}
\arguments{
\item{x}{Network input. Can be:
\itemize{
\item A square numeric matrix (adjacency/weight matrix)
\item A data frame with edge list (from, to, optional weight columns)
\item An igraph object
\item A statnet network object
\item A qgraph object
\item A tna object
\item An existing cograph_network object (returned as-is)
}}

\item{directed}{Logical. Force directed interpretation. NULL for auto-detect.}

\item{...}{Additional arguments (currently unused).}
}
\value{
A cograph_network object: a named list with components:
\describe{
\item{\code{from}}{Integer vector of source node indices}
\item{\code{to}}{Integer vector of target node indices}
\item{\code{weight}}{Numeric vector of edge weights}
\item{\code{nodes}}{Data frame with id, label, (x, y if layout applied)}
\item{\code{directed}}{Logical indicating if network is directed}
\item{\code{n_nodes}}{Integer count of nodes}
\item{\code{n_edges}}{Integer count of edges}
\item{\code{labels}}{Character vector of node labels}
\item{\code{source}}{Character indicating input type}
\item{\code{layout}}{Layout coordinates (NULL until computed)}
\item{\code{layout_info}}{Layout algorithm info (NULL until computed)}
}
}
\description{
Creates a lightweight cograph_network object from various network inputs.
The resulting object is a named list with all data accessible via \code{$}.
}
\details{
The cograph_network format is designed to be:
\itemize{
\item Simple: All data accessible via \code{net$from}, \code{net$to}, \code{net$weight}, etc.
\item Modern: Uses named list elements instead of attributes for clean \code{$} access
\item Compatible: Works seamlessly with splot() and other cograph functions
}

Use getter functions for programmatic access:
\code{\link{get_nodes}}, \code{\link{get_edges}}, \code{\link{get_labels}}

Use setter functions to modify:
\code{\link{set_nodes}}, \code{\link{set_edges}}, \code{\link{set_layout}}
}
\examples{
# From adjacency matrix
mat <- matrix(c(0, 1, 1, 1, 0, 1, 1, 1, 0), nrow = 3)
net <- as_cograph(mat)

# Direct $ access to all data
net$from       # edge sources
net$to         # edge targets
net$weight     # edge weights
net$nodes      # nodes data frame
net$directed   # TRUE/FALSE
net$n_nodes    # 3
net$n_edges    # 3

# Getter functions (recommended for programmatic use)
get_nodes(net)   # nodes data frame
get_edges(net)   # edges data frame (from, to, weight)
get_labels(net)  # character vector of labels
n_nodes(net)     # 3
n_edges(net)     # 3
is_directed(net) # FALSE (symmetric matrix)

# Setter functions
net <- set_nodes(net, data.frame(id = 1:3, label = c("A", "B", "C")))
net <- set_edges(net, data.frame(from = c(1,2), to = c(2,3), weight = c(0.5, 0.8)))
net <- set_layout(net, data.frame(x = c(0, 1, 0.5), y = c(0, 0, 1)))

# Plot it
splot(net)

# From igraph (if installed)
if (requireNamespace("igraph", quietly = TRUE)) {
  library(igraph)
  g <- make_ring(10)
  net <- as_cograph(g)
  splot(net)
}
}
\seealso{
\code{\link{get_nodes}} to extract the nodes data frame,
\code{\link{get_edges}} to extract edges as a data frame,
\code{\link{n_nodes}} and \code{\link{n_edges}} for counts,
\code{\link{is_directed}} to check directedness,
\code{\link{splot}} for plotting
}
