% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/outcome_models.R
\name{outcome_continuous}
\alias{outcome_continuous}
\title{Simulate from continuous outcome model given covariates}
\usage{
outcome_continuous(
  data,
  mean = NULL,
  par = NULL,
  sd = 1,
  het = 0,
  outcome.name = "y",
  remove = c("id", "num"),
  family = gaussian(),
  ...
)
}
\arguments{
\item{data}{(data.table) Covariate data, usually the output of the covariate
model of a \link{Trial} object.}

\item{mean}{(formula, function) Either a formula specifying the design from
'data' or a function that maps \code{data} to the conditional mean value on the
link scale (see examples). If NULL all main-effects of the covariates will
be used, except columns that are defined via the \code{remove} argument.}

\item{par}{(numeric) Regression coefficients (default zero). Can be given as
a named list corresponding to the column names of \code{model.matrix}}

\item{sd}{(numeric) standard deviation of Gaussian measurement error}

\item{het}{Introduce variance hetereogeneity by adding a residual term
\eqn{het \cdot \mu_x \cdot e}, where \eqn{\mu_x} is the mean given
covariates and \eqn{e} is an independent standard normal distributed
variable. This term is in addition to the measurement error introduced by
the \code{sd} argument.}

\item{outcome.name}{Name of outcome variable ("y")}

\item{remove}{Variables that will be removed from input \code{data} (if formula is
not specified).}

\item{family}{exponential family (default \code{gaussian(identity)})}

\item{...}{Additional arguments passed to \code{mean} function (see examples)}
}
\value{
data.table
}
\description{
Simulate from continuous outcome model with mean
\deqn{g(\text{par}^\top X)} where \eqn{X} is the design matrix specified by
the formula, and \eqn{g} is the link function specified by the family
argument
}
\examples{
trial <- Trial$new(
  covariates = \(n) data.frame(a = rbinom(n, 1, 0.5), x = rnorm(n)),
  outcome = outcome_continuous
)
est <- function(data) glm(y ~ a + x, data = data)
trial$simulate(1e4, mean = ~ 1 + a + x, par = c(1, 0.5, 2)) |> est()

# default behavior is to set all regression coefficients to 0
trial$simulate(1e4, mean = ~ 1 + a + x) |> est()

# intercept defaults to 0 and regression coef for a takes the provided value
trial$simulate(1e4, mean = ~ 1 + a, par = c(a = 0.5)) |> est()
# trial$simulate(1e4, mean = ~ 1 + a, par = c("(Intercept)" = 0.5)) |> est()

# define mean model that directly works on whole covariate data, incl id and
# num columns
trial$simulate(1e4, mean = \(x) with(x, -1 + a * 2 + x * -3)) |>
  est()

# par argument is not passed on to mean function
trial$simulate(1e4,
  mean = \(x,  reg.par) with(x, reg.par[1] + reg.par[2] * a),
  reg.par = c(1, 5)
) |> est()
}
\seealso{
\link{outcome_count} \link{outcome_binary}
}
