% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dissimilarity.R
\name{dissimilarity}
\alias{dissimilarity}
\title{Compute dissimilarity metrics (beta-diversity) between sites based on
species composition}
\usage{
dissimilarity(comat, metric = "Simpson", formula = NULL, method = "prodmat")
}
\arguments{
\item{comat}{A co-occurrence \code{matrix} with sites as rows and species as
columns.}

\item{metric}{A \code{character} vector or a single \code{character} string specifying
the metrics to compute (see Details). Available options are \code{"abc"}, \code{"ABC"},
\code{"Jaccard"}, \code{"Jaccardturn"}, \code{"Sorensen"}, \code{"Simpson"}, \code{"Bray"},
\code{"Brayturn"}, and \code{"Euclidean"}. If \code{"all"} is specified, all metrics will
be calculated. Can be set to \code{NULL} if \code{formula} is used.}

\item{formula}{A \code{character} vector or a single \code{character} string specifying
custom formula(s) based on the \code{a}, \code{b}, \code{c}, \code{A}, \code{B}, and \code{C} quantities
(see Details). The default is \code{NULL}.}

\item{method}{A \code{character} string specifying the method to compute \code{abc}
(see Details). The default is \code{"prodmat"}, which is more efficient but
memory-intensive. Alternatively, \code{"loops"} is less memory-intensive but
slower.}
}
\value{
A \code{data.frame} with the additional class
\code{bioregion.pairwise}, containing one or several dissimilarity
metrics between pairs of sites. The first two columns represent the pairs of
sites. There is one column per similarity metric provided in \code{metric} and
\code{formula}, except for the \code{abc} and \code{ABC} metrics, which are stored in three
separate columns (one for each letter).
}
\description{
This function generates a \code{data.frame} where each row provides one or
several dissimilarity metrics between pairs of sites, based on a
co-occurrence \code{matrix} with sites as rows and species as columns.
}
\details{
With \code{a} the number of species shared by a pair of sites, \code{b} species only
present in the first site  and \code{c} species only present in the second site.

Jaccard = (b + c) / (a + b + c)

Jaccardturn = 2min(b, c) / (a + 2min(b, c)) (Baselga, 2012)

Sorensen = (b + c) / (2a + b + c)

Simpson = min(b, c) / (a + min(b, c))

If abundances data are available, Bray-Curtis and its turnover component
can also be computed with the following equation:

Bray = (B + C) / (2A + B + C)

Brayturn = min(B, C)/(A + min(B, C)) (Baselga, 2013)

with \code{A} the sum of the lesser values for common species shared by a pair of
sites. \code{B} and \code{C} are the total number of specimens counted at both sites
minus \code{A}.

\code{formula} can be used to compute customized metrics with the terms
\code{a}, \code{b}, \code{c}, \code{A}, \code{B}, and \code{C}. For example
\code{formula = c("pmin(b,c) / (a + pmin(b,c))", "(B + C) / (2*A + B + C)")}
will compute the Simpson and Bray-Curtis dissimilarity metrics, respectively.
Note that \code{pmin} is used in the Simpson formula because \code{a}, \code{b}, \code{c}, \code{A},
\code{B} and \code{C} are \code{numeric} vectors.

Euclidean computes the Euclidean distance between each pair of sites.
}
\examples{
comat <- matrix(sample(0:1000, size = 50, replace = TRUE,
prob = 1 / 1:1001), 5, 10)
rownames(comat) <- paste0("s", 1:5)
colnames(comat) <- paste0("sp", 1:10)

dissim <- dissimilarity(comat,
metric = c("abc", "ABC", "Simpson", "Brayturn"))

dissim <- dissimilarity(comat, metric = "all",
formula = "1 - (b + c) / (a + b + c)")

}
\references{
Baselga, A. (2012) The Relationship between Species Replacement,
Dissimilarity Derived from Nestedness, and Nestedness.
\emph{Global Ecology and Biogeography}, 21(12), 1223--1232.

Baselga, A. (2013) Separating the two components of abundance-based
dissimilarity: balanced changes in abundance vs. abundance gradients.
\emph{Methods in Ecology and Evolution}, 4(6), 552--557.
}
\seealso{
For more details illustrated with a practical example,
see the vignette:
\url{https://biorgeo.github.io/bioregion/articles/a3_pairwise_metrics.html}.

Associated functions:
\link{similarity} \link{dissimilarity_to_similarity}
}
\author{
Maxime Lenormand (\email{maxime.lenormand@inrae.fr}) \cr
Pierre Denelle (\email{pierre.denelle@gmail.com}) \cr
Boris Leroy (\email{leroy.boris@gmail.com})
}
