% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mice.spark.R
\name{mice.spark}
\alias{mice.spark}
\title{MICE for Spark DataFrames using Sparklyr and Spark MLlib}
\usage{
mice.spark(
  data,
  sc,
  variable_types,
  analysis_formula,
  m = 5,
  method = NULL,
  predictorMatrix = NULL,
  formulas = NULL,
  modeltype = NULL,
  maxit = 5,
  printFlag = TRUE,
  seed = NA,
  imp_init = NULL,
  checkpointing = TRUE,
  checkpoint_frequency = 10,
  ...
)
}
\arguments{
\item{data}{A Spark DataFrame}

\item{sc}{A Spark connection}

\item{variable_types}{A named character vector, the variable types of the columns in the data.}

\item{analysis_formula}{A formula, the formula to use for the analysis}

\item{m}{The number of imputations to perform}

\item{method}{A character vector, the imputation method to use for each variable. If NULL, the function will infer the method based on the variable types.}

\item{predictorMatrix}{A matrix, the predictor matrix to use for the imputation.}

\item{formulas}{A list, the formulas to use for the imputation. If NULL, the function will infer the formulas based on the other variables present in the data.}

\item{modeltype}{A character vector, the model type to use for the imputation. If NULL, the function will infer the model type based on the variable types.}

\item{maxit}{The maximum number of iterations to perform}

\item{printFlag}{A boolean, whether to print debug information}

\item{seed}{An integer, the seed to use for reproducibility}

\item{imp_init}{A Spark DataFrame, the original data with missing values, but with initial imputation (by random sampling or mean/median/mode imputation). Can be set to avoid re-running the initialisation step. Otherwise, the function will perform the initialisation step using the MeMoMe function.}

\item{checkpointing}{Default TRUE. Can be set to FALSE if you are running the package without access to a HDFS directory for checkpointing. It is strongly recommended to keep it to TRUE to avoid Stackoverflow errors.}

\item{checkpoint_frequency}{Advanced parameter, modify with care. If checkpointing = TRUE, how often to checkpoint , default = 10, so after processing every 10 variables, the lineage will be cut and the current state of computation will be save to disk. A low number might slow down computation but enable bigger computation. A number too high (or not checkpoiting) might cause JVM stackOverflowError as the lineage will have grown too big.}

\item{...}{Additional arguments to be passed to the function.}
}
\value{
A list containing the Rubin's statistics for the model parameters, the per-imputation statistics, the imputation statistics, and the model parameters.
}
\description{
This function imputes missing values in a Spark DataFrame using MICE (Multiple Imputation by Chained Equations) algorithm.
}
\examples{
# This example is not executed since it needs additional software (Apache Spark)
\dontrun{
# Example for mice.spark function
library(sparklyr)
library(dplyr)

# Connect to Spark
# Assumes that you have already installed Spark with sparklyr::spark_install()
sc <- spark_connect(master = "local")

# Create sample data with missing values
sample_data <- data.frame(
  outcome = c(1, 0, NA, 1, NA, 0),
  age = c(25, NA, 35, 28, 45, NA),
  income = c(50000, 60000, NA, 55000, 80000, 52000),
  education = c("High", "Medium", "High", NA, "Medium", "Medium")
)

# Copy to Spark DataFrame
sdf <- copy_to(sc, sample_data, "sample_data")

# Define variable types
variable_types <- c(
  outcome = "Binary",
  age = "Continuous_int",
  income = "Continuous_int",
  education = "Nominal"
)

# Define analysis formula
analysis_formula <- outcome ~ age + income + education

# Run MICE imputation
mice_result <- mice.spark(
  data = sdf,
  sc = sc,
  variable_types = variable_types,
  analysis_formula = analysis_formula,
  m = 3,  # Number of imputations
  maxit = 2,  # Number of iterations
  printFlag = TRUE,
  seed = 123,
  checkpointing = FALSE  # Set to TRUE if HDFS is available
)

# See results
 print(mice_result)
# Clean up
 spark_disconnect(sc)
 }
}
\references{
Morvan H, Agholme J, Eliasson B, Olofsson K, Grote L, Iredahl F, Sysoev O.
bigMICE: Multiple Imputation of Big Data. arXiv:2601.21613 [stat.CO]. 2026.
\doi{10.48550/arXiv.2601.21613}
}
