% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hurdle-demand.R
\name{fit_demand_hurdle}
\alias{fit_demand_hurdle}
\title{Fit Two-Part Mixed Effects Hurdle Demand Model}
\usage{
fit_demand_hurdle(
  data,
  y_var,
  x_var,
  id_var,
  random_effects = c("zeros", "q0", "alpha"),
  epsilon = 0.001,
  start_values = NULL,
  tmb_control = list(max_iter = 200, eval_max = 1000, trace = 0),
  verbose = 1,
  part2 = c("zhao_exponential", "exponential", "simplified_exponential"),
  ...
)
}
\arguments{
\item{data}{A data frame containing the demand data.}

\item{y_var}{Character string specifying the column name for consumption values.}

\item{x_var}{Character string specifying the column name for price.}

\item{id_var}{Character string specifying the column name for subject IDs.}

\item{random_effects}{Character vector specifying which random effects to include.
Options are \code{"zeros"} (a_i for Part I), \code{"q0"} (b_i for intensity),
and \code{"alpha"} (c_i for elasticity). Default is \code{c("zeros", "q0", "alpha")}
for the full 3-random-effect model. Use \code{c("zeros", "q0")} for the
simplified 2-random-effect model (fixed alpha across subjects).}

\item{epsilon}{Small constant added to price before log transformation in Part I.
Used to handle zero prices: \code{log(price + epsilon)}. Default is 0.001.}

\item{start_values}{Optional named list of starting values for optimization.
If \code{NULL} (default), sensible defaults are used.}

\item{tmb_control}{List of control parameters for TMB optimization:
\describe{
\item{max_iter}{Maximum number of optimization iterations (default 200)}
\item{eval_max}{Maximum number of function evaluations (default 1000)}
\item{trace}{Print optimization trace: 0 = none, 1 = some (default 0)}
}}

\item{verbose}{Integer controlling output verbosity: 0 = silent, 1 = progress
messages, 2 = detailed optimization trace. Default is 1.}

\item{part2}{Character string selecting the Part II mean function. Options are
\code{"zhao_exponential"} (default; no Q0 normalization in the exponent),
\code{"exponential"} (HS-standardized; Q0 inside the exponent), and
\code{"simplified_exponential"} (SND/log-linear; no \code{k} parameter).}

\item{...}{Additional arguments (reserved for future use).}
}
\value{
An object of class \code{beezdemand_hurdle} containing:
\describe{
\item{model}{List with coefficients, se, variance_components, correlations}
\item{random_effects}{Matrix of empirical Bayes random effect estimates}
\item{subject_pars}{Data frame of subject-specific parameters including
Q0, alpha, breakpoint, Pmax, Omax}
\item{tmb_obj}{TMB objective function object}
\item{opt}{Optimization result from \code{nlminb}}
\item{sdr}{TMB sdreport object}
\item{call}{The matched call}
\item{data}{Original data used for fitting}
\item{param_info}{List with y_var, x_var, id_var, n_subjects, n_obs, etc.}
\item{converged}{Logical indicating convergence}
\item{loglik}{Log-likelihood at convergence}
\item{AIC, BIC}{Information criteria}
\item{error_message}{Error message if fitting failed, NULL otherwise}
}
}
\description{
Fits a two-part hurdle model for demand data using TMB (Template Model Builder).
Part I models the probability of zero consumption using logistic regression.
Part II models log-consumption given positive response using a nonlinear
mixed effects model.
}
\details{
The model structure is:

\strong{Part I (Binary - probability of zero consumption):}
\deqn{logit(\pi_{ij}) = \beta_0 + \beta_1 \cdot \log(price + \epsilon) + a_i}

\strong{Part II (Continuous - log consumption given positive):}

With 3 random effects (\code{random_effects = c("zeros", "q0", "alpha")}):
\deqn{\log(Q_{ij}) = (\log Q_0 + b_i) + k \cdot (\exp(-\alpha_i \cdot price) - 1) + \epsilon_{ij}}
where \eqn{\alpha_i = \exp(\log(\alpha) + c_i)} and \eqn{k = \exp(\log(k))}.

With 2 random effects (\code{random_effects = c("zeros", "q0")}):
\deqn{\log(Q_{ij}) = (\log Q_0 + b_i) + k \cdot (\exp(-\alpha \cdot price) - 1) + \epsilon_{ij}}
where \eqn{\alpha = \exp(\log(\alpha))} and \eqn{k = \exp(\log(k))}.

Random effects follow a multivariate normal distribution with unstructured
covariance matrix. Use \code{\link{compare_hurdle_models}} for likelihood
ratio tests comparing nested models.
}
\section{Parameterization and comparability}{

The TMB backend estimates positive-constrained parameters on the natural-log
scale: \eqn{\log(Q_0)}, \eqn{\log(\alpha)}, and \eqn{\log(k)}. Reporting methods
(\code{summary()}, \code{tidy()}, \code{coef()}) can back-transform to the natural scale or
present parameters on the \eqn{\log_{10}} scale.

To compare \eqn{\alpha} estimates with models fit in \eqn{\log_{10}} space,
use:
\deqn{\log_{10}(\alpha) = \log(\alpha) / \log(10).}
}

\examples{
\donttest{
# Load example data
data(apt)

# Fit full model with 3 random effects
fit3 <- fit_demand_hurdle(apt, y_var = "y", x_var = "x", id_var = "id",
                          random_effects = c("zeros", "q0", "alpha"))

# Fit simplified model with 2 random effects (fixed alpha)
fit2 <- fit_demand_hurdle(apt, y_var = "y", x_var = "x", id_var = "id",
                          random_effects = c("zeros", "q0"))

# View results
summary(fit3)

# Compare models with likelihood ratio test
compare_hurdle_models(fit3, fit2)
}

}
\seealso{
\code{\link{summary.beezdemand_hurdle}}, \code{\link{predict.beezdemand_hurdle}},
\code{\link{plot.beezdemand_hurdle}}, \code{\link{compare_hurdle_models}},
\code{\link{simulate_hurdle_data}}
}
