% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cross-price.R
\name{fit_cp_nls}
\alias{fit_cp_nls}
\title{Fit cross-price demand with NLS (+ robust fallbacks)}
\usage{
fit_cp_nls(
  data,
  equation = c("exponentiated", "exponential", "additive"),
  x_var = "x",
  y_var = "y",
  start_values = NULL,
  start_vals = lifecycle::deprecated(),
  iter = 100,
  bounds = NULL,
  fallback_to_nlsr = TRUE,
  return_all = TRUE
)
}
\arguments{
\item{data}{A data frame with columns for price and consumption.
Additional columns are ignored. Input is validated internally.}

\item{equation}{Character string; model family, one of
\code{c("exponentiated", "exponential", "additive")}. Default is \code{"exponentiated"}.}

\item{x_var}{Character string; name of the price column in \code{data}. Default is
\code{"x"}. The column is renamed to \code{"x"} internally; see \code{validate_cp_data()}.}

\item{y_var}{Character string; name of the consumption column in \code{data}. Default
is \code{"y"}. The column is renamed to \code{"y"} internally.}

\item{start_values}{Optional \strong{named list} of initial values for parameters
\code{log10_qalone}, \code{I}, and \code{log10_beta}. If \code{NULL}, the function derives
plausible ranges from the data and uses multi-start search.}

\item{start_vals}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use \code{start_values} instead.}

\item{iter}{Integer; number of random starts for \code{nls.multstart} (default \code{100}).}

\item{bounds}{Deprecated. Log10-parameterized parameters are naturally unbounded.
This argument is ignored but retained for backwards compatibility.}

\item{fallback_to_nlsr}{Logical; if \code{TRUE} (default), try \code{nlsr::wrapnlsr} when
both multi-start NLS and \code{nlsLM} fail.}

\item{return_all}{Logical; if \code{TRUE} (default), return a list containing the
model and useful metadata. If \code{FALSE}, return the bare fitted model object.}
}
\value{
If \code{return_all = TRUE} (default): a list of class \code{"cp_model_nls"}:
\itemize{
\item \code{model}: the fitted object from the successful backend.
\item \code{method}: one of \code{"nls_multstart"}, \code{"nlsLM"}, or \code{"wrapnlsr"}.
\item \code{equation}: the model family used.
\item \code{start_vals}: named list of starting values (final used; kept for backward compatibility).
\item \code{nlsLM_fit}, \code{nlsr_fit}: fits from later stages (if attempted).
\item \code{data}: the 2-column data frame actually fit.
}
If \code{return_all = FALSE}: the fitted model object from the successful backend.
}
\description{
Fits a \strong{cross-price demand} curve using log10-parameterization for numerical
stability. The optimizer estimates parameters on the log10 scale where applicable,
ensuring positive constraints are naturally satisfied.

\strong{Equation forms:}
\itemize{
\item \strong{Exponentiated} (default):
\deqn{y = Q_{alone} \cdot 10^{I \cdot \exp(-\beta \cdot x)}}
\item \strong{Exponential} (fits on log10 response scale):
\deqn{\log_{10}(y) = \log_{10}(Q_{alone}) + I \cdot \exp(-\beta \cdot x)}
\item \strong{Additive} (level on \eqn{y}):
\deqn{y = Q_{alone} + I \cdot \exp(-\beta \cdot x)}
}

where \eqn{x} is the alternative product price (or "cross" price) and \eqn{y}
is consumption of the target good.

\strong{Optimizer parameters (log10 parameterization):}
\itemize{
\item \code{log10_qalone}: \eqn{\log_{10}(Q_{alone})} - baseline consumption
when the alternative is effectively absent.
\item \code{I}: cross-price \strong{interaction intensity}; sign and magnitude reflect
substitution/complementarity. Unconstrained (can be negative for substitutes).
\item \code{log10_beta}: \eqn{\log_{10}(\beta)} - rate at which cross-price
influence decays as \eqn{x} increases.
}

Natural-scale values are recovered as \eqn{Q_{alone} = 10^{log10\_qalone}} and
\eqn{\beta = 10^{log10\_beta}}.

The function first attempts a multi-start nonlinear least squares fit
(\code{nls.multstart}). If that fails—or if explicit \code{start_values} are provided—it
falls back to \code{minpack.lm::nlsLM}. Optionally, it will make a final attempt
with \code{nlsr::wrapnlsr}. Returns either the fitted model or a structured object
with metadata for downstream methods.
}
\details{
\strong{Start values.} When \code{start_values} is missing, the function:
(1) estimates a reasonable range for \code{log10_qalone} from the observed \code{y},
(2) estimates \code{log10_beta} from the price range, and (3) launches a multi-start
grid in \code{nls.multstart}.

\strong{Zero handling for exponential equation.} Since the exponential equation fits
on the \eqn{\log_{10}(y)} scale, observations with \eqn{y \le 0} are automatically
removed with a warning. Use the exponentiated or additive forms if you need to
retain zero consumption values.

\strong{Fitting pipeline (short-circuiting):}
\enumerate{
\item \code{nls.multstart::nls_multstart()} with random starts.
\item If that fails (or if \code{start_values} provided): \code{minpack.lm::nlsLM()} using
\code{start_values} (user or internally estimated).
\item If that fails and \code{fallback_to_nlsr = TRUE}: \code{nlsr::wrapnlsr()}.
}

The returned object has class \code{"cp_model_nls"} (when \code{return_all = TRUE}) with
components: \code{model}, \code{method} (the algorithm used), \code{equation}, \code{start_vals},
\code{nlsLM_fit}, \code{nlsr_fit}, and the \code{data} used. This is convenient for custom
print/summary/plot methods.
}
\section{Convergence & warnings}{

\itemize{
\item Check convergence codes and residual diagnostics from the underlying fit.
\item Poor scaling or extreme \code{y} dispersion can make parameters weakly identified.
\item For \code{"exponential"}, the model fits on the \eqn{\log_{10}(y)} scale internally.
}
}

\examples{
## --- Example: Real data (E-Cigarettes, id = 1) ---
dat <- structure(list(
  id = c(1, 1, 1, 1, 1, 1),
  x = c(2, 4, 8, 16, 32, 64),
  y = c(3, 5, 5, 16, 17, 13),
  target = c("alt", "alt", "alt", "alt", "alt", "alt"),
  group = c("E-Cigarettes", "E-Cigarettes", "E-Cigarettes",
            "E-Cigarettes", "E-Cigarettes", "E-Cigarettes")
), row.names = c(NA, -6L), class = c("tbl_df", "tbl", "data.frame"))

## Fit the default (exponentiated) cross-price form
fit_ecig <- fit_cp_nls(dat, equation = "exponentiated", return_all = TRUE)
summary(fit_ecig)         # model summary
fit_ecig$method           # backend actually used (e.g., "nls_multstart")
coef(fit_ecig$model)      # parameter estimates: log10_qalone, I, log10_beta
}
\seealso{
\code{\link{check_unsystematic_cp}} for pre-fit data screening,
\code{\link{validate_cp_data}} for input validation.
}
