% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/baselinenowcast.R
\name{baselinenowcast.reporting_triangle}
\alias{baselinenowcast.reporting_triangle}
\title{Create a dataframe of nowcast results from a single reporting triangle}
\usage{
\method{baselinenowcast}{reporting_triangle}(
  data,
  scale_factor = 3,
  prop_delay = 0.5,
  output_type = c("samples", "point"),
  draws = 1000,
  uncertainty_model = fit_by_horizon,
  uncertainty_sampler = sample_nb,
  delay_pmf = NULL,
  uncertainty_params = NULL,
  preprocess = preprocess_negative_values,
  validate = TRUE,
  ...
)
}
\arguments{
\item{data}{\link{reporting_triangle} class object to be nowcasted.
The matrix must contain missing observations in the form of NAs in order
to generate an output from this function.}

\item{scale_factor}{Numeric value indicating the multiplicative factor on
the maximum delay to be used for estimation of delay and uncertainty.
Default is \code{3}.}

\item{prop_delay}{Numeric value <1 indicating what proportion of all
reference times in the reporting triangle to be used for delay
estimation. Default is \code{0.5}.}

\item{output_type}{Character string indicating whether the output should be
samples (\code{"samples"}) from the estimate with full uncertainty or whether to
return the point estimate (\code{"point"}). Default is \code{"samples"}. If
\code{"point"}estimates are specified, the minimum number of reference times
needed is the number needed for delay estimation, otherwise, if
\code{"samples"} are specified, at least 2 additional reference times are
required for uncertainty estimation.}

\item{draws}{Integer indicating the number of probabilistic draws to include
if \code{output_type} is \code{"samples"}. Default is 1000.}

\item{uncertainty_model}{Function that ingests a matrix of observations and a
matrix of predictions and returns a vector that can be used to
apply uncertainty using the same error model. Default is
\code{fit_by_horizon} with arguments of \code{obs} matrix of observations and
\code{pred} the matrix of predictions that fits each column (horizon)
to a negative binomial observation model by default. The user can
specify a different fitting model by replacing the
\code{fit_model} argument in \code{fit_by_horizon}.}

\item{uncertainty_sampler}{Function that ingests a vector or matrix of
predictions and a vector of uncertainty parameters and generates draws
from the observation model. Default is \code{sample_nb} which expects
arguments \code{pred} for the vector of predictions and uncertainty parameters
for the corresponding vector of uncertainty parameters, and draws from a
negative binomial for each element of the vector.}

\item{delay_pmf}{Vector of delays assumed to be indexed starting at the
first delay column in the reporting triangle. Default is NULL, which will
estimate the delay from the reporting triangle in \code{data}. See
\code{\link[=estimate_delay]{estimate_delay()}} for more details.}

\item{uncertainty_params}{Vector of uncertainty parameters ordered from
horizon 1 to the maximum horizon. Default is \code{NULL}, which will result in
computing the uncertainty parameters from the reporting triangle \code{data}.
See \code{\link[=estimate_uncertainty]{estimate_uncertainty()}} for more details.}

\item{preprocess}{Function to apply to the reporting triangle before
estimation, or NULL to skip preprocessing. Default is
\code{\link[=preprocess_negative_values]{preprocess_negative_values()}}, which handles negative values by
redistributing them to earlier delays. Set to NULL if you want to preserve
negative values. Custom preprocess functions must accept a \code{validate}
parameter (defaults to TRUE) to enable validation optimisation in internal
function chains.}

\item{validate}{Logical. If TRUE (default), validates the object. Set to
FALSE only when called from functions that already validated.}

\item{...}{Additional arguments passed to
\code{\link[=estimate_uncertainty]{estimate_uncertainty()}}
and \code{\link[=sample_nowcast]{sample_nowcast()}}.}
}
\value{
Data.frame of class \code{\link{baselinenowcast_df}}
}
\description{
This function ingests a single
\link{reporting_triangle} object and generates a nowcast in the
form of a \link{baselinenowcast_df} object.

This function implements a nowcasting workflow for a single reporting
triangle:
\enumerate{
\item \code{\link[=allocate_reference_times]{allocate_reference_times()}} - Allocate the reference times used
for delay and uncertainty estimation
\item \code{\link[=estimate_delay]{estimate_delay()}} - Estimate a reporting delay PMF
\item \code{\link[=apply_delay]{apply_delay()}} - Generate a point nowcast using the delay PMF
\item \code{\link[=estimate_and_apply_uncertainty]{estimate_and_apply_uncertainty()}} - Generate a probabilistic
nowcast from a point nowcast and reporting triangle
}

This function will by default estimate the delay from the
\link{reporting_triangle} and estimate uncertainty using past
retrospective nowcast errors on that \link{reporting_triangle} to
generate probabilistic nowcasts, which
are samples from the predictive distribution of the estimated final case
count at each reference date.
Alternatives include passing in a separate \code{delay_pmf} or
\code{uncertainty_params}.
This method specifically computes a nowcast for a single reporting
triangle. See documentation for the arguments of this function which
can be used to set the model specifications (things like number of
reference times for delay and uncertainty estimation, the observation
model, etc.).
}
\examples{
# Filter to recent data and truncate to reasonable max_delay for faster
# example
data_as_of_df <- syn_nssp_df[syn_nssp_df$report_date <= "2026-04-01", ]
rep_tri <- as_reporting_triangle(data = data_as_of_df) |>
  truncate_to_delay(max_delay = 25) |>
  tail(n = 40)
nowcast_df <- baselinenowcast(rep_tri, draws = 100)
nowcast_df
}
\seealso{
Main nowcasting interface functions
\code{\link{assert_baselinenowcast_df}()},
\code{\link{baselinenowcast}()},
\code{\link{baselinenowcast.data.frame}()},
\code{\link{baselinenowcast_df-class}},
\code{\link{new_baselinenowcast_df}()}
}
\concept{baselinenowcast_df}
