% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pb_reg.R
\name{pb_reg}
\alias{pb_reg}
\title{Passing-Bablok Regression for Method Comparison}
\usage{
pb_reg(
  formula,
  data,
  id = NULL,
  method = c("scissors", "symmetric", "invariant"),
  conf.level = 0.95,
  weights = NULL,
  error.ratio = 1,
  replicates = 0,
  model = TRUE,
  keep_data = TRUE,
  ...
)
}
\arguments{
\item{formula}{A formula of the form \code{y ~ x} specifying the model.}

\item{data}{Data frame with all data.}

\item{id}{Column with subject identifier (optional). If provided, measurement error
ratio is calculated from replicate measurements.}

\item{method}{Method for Passing-Bablok estimation. Options are:
\itemize{
\item "scissors": Scissors estimator (1988) - most robust, scale invariant (default)
\item "symmetric": Original Passing-Bablok (1983) - symmetric around 45-degree line
\item "invariant": Scale-invariant method (1984) - adaptive reference line
}}

\item{conf.level}{The confidence level required. Default is 95\%.}

\item{weights}{An optional vector of case weights to be used in the fitting process.
Should be NULL or a numeric vector.}

\item{error.ratio}{Ratio of measurement error variances (var(x)/var(y)). Default is 1.
This argument is ignored if subject identifiers are provided via \code{id}.}

\item{replicates}{Number of bootstrap iterations for confidence intervals. If 0 (default),
analytical confidence intervals are used. Bootstrap is recommended for weighted
data and 'invariant' or 'scissors' methods.}

\item{model}{Logical. If TRUE (default), the model frame is stored in the returned object.
This is needed for methods like \code{plot()}, \code{fitted()}, \code{residuals()}, and \code{predict()} to work
without supplying \code{data}. If FALSE, the model frame is not stored (saves memory for large datasets),
but these methods will require a \code{data} argument.}

\item{keep_data}{Logical indicator (TRUE/FALSE). If TRUE, intermediate calculations
are returned; default is FALSE.}

\item{...}{Additional arguments (currently unused).}
}
\value{
The function returns a simple_eiv object with the following components:
\itemize{
\item \code{coefficients}: Named vector of coefficients (intercept and slope).
\item \code{residuals}: Residuals from the fitted model.
\item \code{fitted.values}: Predicted Y values.
\item \code{model_table}: Data frame presenting the full results from the Passing-Bablok regression.
\item \code{vcov}: Variance-covariance matrix for slope and intercept (if bootstrap used).
\item \code{df.residual}: Residual degrees of freedom.
\item \code{call}: The matched call.
\item \code{terms}: The terms object used.
\item \code{model}: The model frame.
\item \code{x_vals}: Original x values used in fitting.
\item \code{y_vals}: Original y values used in fitting.
\item \code{weights}: Case weights (if provided).
\item \code{error.ratio}: Error ratio used in fitting.
\item \code{conf.level}: Confidence level used.
\item \code{method}: Character string describing the method.
\item \code{method_num}: Numeric method identifier (1, 2, or 3).
\item \code{kendall_test}: Results of Kendall's tau correlation test.
\item \code{cusum_test}: Results of CUSUM linearity test.
\item \code{n_slopes}: Number of slopes used in estimation.
\item \code{boot}: Bootstrap results (if replicates > 0).
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

A robust, nonparametric method for fitting a straight line to two-dimensional data
where both variables (X and Y) are measured with error. Particularly useful for
method comparison studies.
}
\details{
Passing-Bablok regression is a robust nonparametric method that estimates the
slope as the shifted median of all possible slopes between pairs of points.
The intercept is then calculated as the median of y - slope*x. This method
is particularly useful when:
\itemize{
\item Both X and Y are measured with error
\item You want a robust method not sensitive to outliers
\item The relationship is assumed to be linear
\item X and Y are highly positively correlated
}
\subsection{Methods}{

Three Passing-Bablok methods are available:

\strong{"scissors"} (default): The scissors estimator (1988), most robust and
scale-invariant. Uses the median of absolute values of angles.

\strong{"symmetric"}: The original method (1983), symmetric about the y = x line.
Uses the line y = -x as the reference for partitioning points.

\strong{"invariant"}: Scale-invariant method (1984). First finds the median angle
of slopes below the horizontal, then uses this as the reference line.
}

\subsection{Measurement Error Handling}{

If the data are measured in replicates, then the measurement error ratio can be
directly derived from the data. This can be accomplished by indicating the subject
identifier with the \code{id} argument. When replicates are not available in the data,
then the ratio of error variances (var(x)/var(y)) can be provided with the
\code{error.ratio} argument (default = 1, indicating equal measurement errors).

The error ratio affects how pairwise slopes are weighted in the robust median
calculation. When error.ratio = 1, all pairs receive equal weight. When
error.ratio != 1, pairs are weighted to account for heterogeneous measurement
precision.
}

\subsection{Weighting}{

Case weights can be provided via the \code{weights} argument. These are distinct from
measurement error weighting (controlled by \code{error.ratio}). Case weights allow
you to down-weight or up-weight specific observations in the analysis.
}

\subsection{Bootstrap}{

Wild bootstrap resampling is used when \code{replicates > 0}. This is particularly
useful for:
\itemize{
\item Weighted regression (case weights or error.ratio != 1)
\item Methods 'invariant' and 'scissors' (where analytical CI validity is uncertain)
\item Small sample sizes
}

The method automatically:
\itemize{
\item Tests for high positive correlation using Kendall's tau
\item Tests for linearity using a CUSUM test
\item Computes confidence intervals (analytical or bootstrap)
}
}
}
\examples{
\dontrun{
# Basic Passing-Bablok regression (scissors method, default)
model <- pb_reg(method2 ~ method1, data = mydata)

# With known error ratio
model_er <- pb_reg(method2 ~ method1, data = mydata, error.ratio = 2)

# With replicate measurements
model_rep <- pb_reg(method2 ~ method1, data = mydata, id = "subject_id")

# With bootstrap confidence intervals
model_boot <- pb_reg(method2 ~ method1, data = mydata,
                     error.ratio = 1.5, replicates = 1000)

# Symmetric method
model_sym <- pb_reg(method2 ~ method1, data = mydata, method = "symmetric")

# Scale-invariant method
model_inv <- pb_reg(method2 ~ method1, data = mydata, method = "invariant")

# With case weights
model_wt <- pb_reg(method2 ~ method1, data = mydata,
                   weights = mydata$case_weights)

# View results
print(model)
summary(model)
plot(model)
}

}
\references{
Passing, H. and Bablok, W. (1983). A new biometrical procedure for testing the
equality of measurements from two different analytical methods. Journal of
Clinical Chemistry and Clinical Biochemistry, 21, 709-720.

Passing, H. and Bablok, W. (1984). Comparison of several regression procedures
for method comparison studies and determination of sample sizes. Journal of
Clinical Chemistry and Clinical Biochemistry, 22, 431-445.

Bablok, W., Passing, H., Bender, R. and Schneider, B. (1988). A general
regression procedure for method transformation. Journal of Clinical Chemistry
and Clinical Biochemistry, 26, 783-790.
}
