#' @title Yield Stability Index (YSI)
#' @description
#' This function computes Yield Stability Index (YSI) using any traits (like yield) under stress and non-stress conditions. The lower values of YSI indicates greater tolerance. For more details see Bouslama and Schapaugh (1984) <doi:10.2135/cropsci1984.0011183X002400050026x>.
#'
#' @param Gen Character vector of genotype identifiers.
#' @param YN Numeric vector: yield (any trait) under non-stress (normal) environment.
#' @param YS Numeric vector: yield (any trait) under stress environment.
#' @return A data frame with columns \code{Gen}, \code{YSI}.
#'
#' \deqn{YSI = \frac{YS}{YN}}
#'
#' @references Bouslama, M. and Schapaugh Jr, W.T. (1984). <doi:10.2135/cropsci1984.0011183X002400050026x>.
#' @examples
#' out = YSI(Gen=c("G1","G2","G3"), YN=c(10,8,5), YS=c(7,5,3))
#' print(out)
#' @export
YSI <- function(Gen, YN, YS) {
  .check_lengths(Gen, YN, YS)
  YN <- as.numeric(YN); YS <- as.numeric(YS)
  out <- YS / YN
  out[YN == 0] <- NA_real_
  return(data.frame(Gen = as.character(Gen), YSI = out, row.names = NULL))
}
