% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/step_obwoe.R
\name{step_obwoe}
\alias{step_obwoe}
\title{Optimal Binning and WoE Transformation Step}
\usage{
step_obwoe(
  recipe,
  ...,
  role = "predictor",
  trained = FALSE,
  outcome = NULL,
  algorithm = "auto",
  min_bins = 2L,
  max_bins = 10L,
  bin_cutoff = 0.05,
  output = c("woe", "bin", "both"),
  suffix_woe = "_woe",
  suffix_bin = "_bin",
  na_woe = 0,
  control = list(),
  binning_results = NULL,
  skip = FALSE,
  id = recipes::rand_id("obwoe")
)
}
\arguments{
\item{recipe}{A recipe object. The step will be added to the sequence of
operations for this recipe.}

\item{...}{One or more selector functions to choose variables for this step.
See \code{\link[recipes]{selections}} for available selectors. Common
choices include \code{all_predictors()}, \code{all_numeric_predictors()},
or \code{all_nominal_predictors()}. Ensure the selected variables are
compatible with the chosen \code{algorithm} (e.g., do not apply
\code{"mdlp"} to categorical data).}

\item{role}{For variables created by this step, what role should they have?

  Default is \code{"predictor"}.}

\item{trained}{A logical indicating whether the step has been trained
(fitted). This should not be set manually.}

\item{outcome}{A character string specifying the name of the binary or
multinomial response variable. This argument is \strong{required} as all
binning algorithms are supervised. The outcome must exist in the training
data provided to \code{prep()}. The outcome should be encoded as a factor
(standard for tidymodels classification) or as integers 0/1 for binary,
0/1/2/... for multinomial.}

\item{algorithm}{Character string specifying the binning algorithm to use.
  Use \code{"auto"} (default) for automatic selection based on target type:
  \code{"jedi"} for binary targets, \code{"jedi_mwoe"} for multinomial.

  Available algorithms are organized by supported feature types:

  \strong{Universal (numerical and categorical):}
  \code{"auto"}, \code{"jedi"}, \code{"jedi_mwoe"}, \code{"cm"}, \code{"dp"},
  \code{"dmiv"}, \code{"fetb"}, \code{"mob"}, \code{"sketch"}, \code{"udt"}

  \strong{Numerical only:}
  \code{"bb"}, \code{"ewb"}, \code{"fast_mdlp"}, \code{"ir"}, \code{"kmb"},
  \code{"ldb"}, \code{"lpdb"}, \code{"mblp"}, \code{"mdlp"}, \code{"mrblp"},
  \code{"oslp"}, \code{"ubsd"}

  \strong{Categorical only:}
  \code{"gmb"}, \code{"ivb"}, \code{"mba"}, \code{"milp"}, \code{"sab"},
  \code{"sblp"}, \code{"swb"}

  This parameter is tunable with \code{tune()}.}

\item{min_bins}{Integer specifying the minimum number of bins to create.
Must be at least 2. Default is 2. This parameter is tunable with
\code{tune()}.}

\item{max_bins}{Integer specifying the maximum number of bins to create.
Must be greater than or equal to \code{min_bins}. Default is 10. This
parameter is tunable with \code{tune()}.}

\item{bin_cutoff}{Numeric value between 0 and 1 (exclusive) specifying the
minimum proportion of total observations that each bin must contain. Bins
with fewer observations are merged with adjacent bins. This serves as a
regularization mechanism to prevent overfitting and ensure statistical
stability of WoE estimates. Default is 0.05 (5\%). This parameter is
tunable with \code{tune()}.}

\item{output}{Character string specifying the transformation output format:
\describe{
  \item{\code{"woe"}}{Replaces the original variable with WoE values
    (default). This is the standard choice for logistic regression
    scorecards.
  }
  \item{\code{"bin"}}{Replaces the original variable with bin labels
    (character). Useful for tree-based models or exploratory analysis.
  }
  \item{\code{"both"}}{Keeps the original column and adds two new columns
    with suffixes \code{_woe} and \code{_bin}. Useful for model comparison
    or audit trails.
  }
}}

\item{suffix_woe}{Character string suffix appended to create WoE column names
when \code{output = "both"}. Default is \code{"_woe"}.}

\item{suffix_bin}{Character string suffix appended to create bin column names
when \code{output = "both"}. Default is \code{"_bin"}.}

\item{na_woe}{Numeric value to assign to observations that cannot be mapped
to a bin during \code{bake()}. This includes missing values (\code{NA}) and
unseen categories not present in the training data. Default is 0, which
represents neutral evidence (neither good nor bad).}

\item{control}{A named list of additional control parameters passed to
\code{\link{control.obwoe}}. These provide fine-grained control over
algorithm behavior such as convergence thresholds and maximum pre-bins.
Parameters specified directly in \code{step_obwoe()} (e.g.,
\code{bin_cutoff}) take precedence over values in this list.}

\item{binning_results}{Internal storage for fitted binning models after
\code{prep()}. Do not set manually.}

\item{skip}{Logical. Should this step be skipped when \code{bake()} is
called on new data? Default is \code{FALSE}. Setting to \code{TRUE} is
rarely needed for WoE transformations but may be useful in specialized
workflows.}

\item{id}{A unique character string to identify this step. If not provided,
a random identifier is generated.}
}
\value{
An updated \code{recipe} object with the new step appended.
}
\description{
\code{step_obwoe()} creates a \emph{specification} of a recipe step that
discretizes predictor variables using one of 28 state-of-the-art optimal
binning algorithms and transforms them into Weight of Evidence (WoE) values.
This step fully integrates the \strong{OptimalBinningWoE} package into the
\code{tidymodels} framework, supporting supervised discretization for both
binary and multinomial classification targets with extensive hyperparameter
tuning capabilities.
}
\details{
\subsection{Weight of Evidence Transformation}{

Weight of Evidence (WoE) is a supervised encoding technique that transforms
categorical and continuous variables into a scale that measures the
predictive strength of each value or bin relative to the target variable.
For a bin \eqn{i}, the WoE is defined as:

\deqn{WoE_i = \ln\left(\frac{\text{Distribution of Events}_i}{\text{Distribution of Non-Events}_i}\right)}

Positive WoE values indicate the bin has a higher proportion of events
(e.g., defaults) than the overall population, while negative values indicate
lower risk.
}

\subsection{Algorithm Selection Strategy}{

The \code{algorithm} parameter provides access to 28 binning algorithms:

\itemize{
  \item Use \code{algorithm = "auto"} (default) for automatic selection:
    \code{"jedi"} for binary targets, \code{"jedi_mwoe"} for multinomial.
  \item Use \code{algorithm = "mob"} (Monotonic Optimal Binning) when
    monotonic WoE trends are required for regulatory compliance (Basel/IFRS 9).
  \item Use \code{algorithm = "mdlp"} for entropy-based discretization of
    numerical variables (requires \code{all_numeric_predictors()}).
  \item Use \code{algorithm = "dp"} (Dynamic Programming) for exact optimal
    solutions when computational cost is acceptable.
}

If an incompatible algorithm is applied to a variable (e.g., \code{"mdlp"}
on a factor), the step will issue a warning during \code{prep()} and skip
that variable, leaving it untransformed.
}

\subsection{Handling New Data}{

During \code{bake()}, observations are mapped to bins learned during
\code{prep()}:

\itemize{
  \item \strong{Numerical variables}: Values are assigned to bins based on
    the learned cutpoints using interval notation.
  \item \strong{Categorical variables}: Categories are matched to their
    corresponding bins. Categories not seen during training receive the
    \code{na_woe} value.
  \item \strong{Missing values}: Always receive the \code{na_woe} value.
}
}

\subsection{Tuning with tune}{

This step is fully compatible with the \code{tune} package. The following
parameters support \code{tune()}:

\itemize{
  \item \code{algorithm}: See \code{\link{obwoe_algorithm}}.
  \item \code{min_bins}: See \code{\link{obwoe_min_bins}}.
  \item \code{max_bins}: See \code{\link{obwoe_max_bins}}.
  \item \code{bin_cutoff}: See \code{\link{obwoe_bin_cutoff}}.
}
}

\subsection{Case Weights}{

This step does not currently support case weights. All observations are
treated with equal weight during binning optimization.
}
}
\examples{
\donttest{
library(recipes)

# Simulated credit data
set.seed(123)
credit_data <- data.frame(
  age = rnorm(500, 45, 12),
  income = exp(rnorm(500, 10, 0.6)),
  employment = sample(c("Employed", "Self-Employed", "Unemployed"),
    500,
    replace = TRUE, prob = c(0.7, 0.2, 0.1)
  ),
  education = factor(c("HighSchool", "Bachelor", "Master", "PhD")[
    sample(1:4, 500, replace = TRUE, prob = c(0.3, 0.4, 0.2, 0.1))
  ]),
  default = factor(rbinom(500, 1, 0.15),
    levels = c(0, 1),
    labels = c("No", "Yes")
  )
)

# Example 1: Basic usage with automatic algorithm selection
rec_basic <- recipe(default ~ ., data = credit_data) \%>\%
  step_obwoe(all_predictors(), outcome = "default")

rec_prepped <- prep(rec_basic)
baked_data <- bake(rec_prepped, new_data = NULL)
head(baked_data)

# View binning details
tidy(rec_prepped, number = 1)

# Example 2: Numerical-only algorithm on numeric predictors
rec_mdlp <- recipe(default ~ age + income, data = credit_data) \%>\%
  step_obwoe(all_numeric_predictors(),
    outcome = "default",
    algorithm = "mdlp",
    min_bins = 3,
    max_bins = 6
  )

# Example 3: Output both bins and WoE
rec_both <- recipe(default ~ age, data = credit_data) \%>\%
  step_obwoe(age,
    outcome = "default",
    output = "both"
  )

baked_both <- bake(prep(rec_both), new_data = NULL)
names(baked_both)
# Contains: default, age, age_woe, age_bin

# Example 4: Custom control parameters
rec_custom <- recipe(default ~ ., data = credit_data) \%>\%
  step_obwoe(all_predictors(),
    outcome = "default",
    algorithm = "mob",
    bin_cutoff = 0.03,
    control = list(
      max_n_prebins = 30,
      convergence_threshold = 1e-8
    )
  )

# Example 5: Tuning specification (for use with tune package)
# rec_tune <- recipe(default ~ ., data = credit_data) \%>\%
#   step_obwoe(all_predictors(),
#              outcome = "default",
#              algorithm = tune(),
#              min_bins = tune(),
#              max_bins = tune())
}

}
\references{
Siddiqi, N. (2006). Credit Risk Scorecards: Developing and Implementing
Intelligent Credit Scoring. \emph{John Wiley & Sons}.
\doi{10.1002/9781119201731}

Thomas, L. C., Edelman, D. B., & Crook, J. N. (2002). Credit Scoring
and Its Applications. \emph{SIAM Monographs on Mathematical Modeling
and Computation}. \doi{10.1137/1.9780898718317}

Navas-Palencia, G. (2020). Optimal Binning: Mathematical Programming
Formulation and Solution Approach. \emph{Expert Systems with Applications},
158, 113508. \doi{10.1016/j.eswa.2020.113508}
}
\seealso{
\code{\link{obwoe}} for the underlying binning engine,
\code{\link{control.obwoe}} for advanced control parameters,
\code{\link{obwoe_algorithm}}, \code{\link{obwoe_min_bins}},
\code{\link{obwoe_max_bins}}, \code{\link{obwoe_bin_cutoff}} for tuning
parameter definitions,
\code{\link[recipes]{recipe}}, \code{\link[recipes]{prep}},
\code{\link[recipes]{bake}} for the tidymodels recipe framework.
}
