% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/obwoe.R
\name{obwoe_apply}
\alias{obwoe_apply}
\title{Apply Weight of Evidence Transformations to New Data}
\usage{
obwoe_apply(
  data,
  obj,
  suffix_bin = "_bin",
  suffix_woe = "_woe",
  keep_original = TRUE,
  na_woe = 0
)
}
\arguments{
\item{data}{A \code{data.frame} containing the features to transform.
Must include all features present in the \code{obj} results. The target
column is optional; if present, it will be included in the output.}

\item{obj}{An object of class \code{"obwoe"} returned by \code{\link{obwoe}}.}

\item{suffix_bin}{Character string suffix for bin columns.
Default is \code{"_bin"}.}

\item{suffix_woe}{Character string suffix for WoE columns.
Default is \code{"_woe"}.}

\item{keep_original}{Logical. If \code{TRUE} (default), include the original
feature columns in the output. If \code{FALSE}, only bin and WoE columns
are returned.}

\item{na_woe}{Numeric value to assign when an observation cannot be mapped
to a bin (e.g., new categories not seen during training). Default is 0.}
}
\value{
A \code{data.frame} containing:
\describe{
  \item{\code{target}}{The target column (if present in \code{data})}
  \item{\code{<feature>}}{Original feature values (if \code{keep_original = TRUE})}
  \item{\code{<feature>_bin}}{Assigned bin label for each observation}
  \item{\code{<feature>_woe}}{Weight of Evidence value for the assigned bin}
}
}
\description{
Applies the binning and Weight of Evidence (WoE) transformations learned by
\code{\link{obwoe}} to new data. This is the scoring function for deploying
WoE-based models in production. For each feature, the function assigns
observations to bins and maps them to their corresponding WoE values.
}
\details{
\subsection{Bin Assignment Logic}{

\strong{Numerical Features}:
Observations are assigned to bins based on cutpoints stored in the
\code{obwoe} object. The \code{cut()} function is used with intervals
\eqn{(a_i, a_{i+1}]} where \eqn{a_0 = -\infty} and \eqn{a_k = +\infty}.

\strong{Categorical Features}:
Categories are matched directly to bin labels. Categories not seen
during training are assigned \code{NA} for bin and \code{na_woe} for WoE.
}

\subsection{Production Deployment}{

For production scoring, it is recommended to:
\enumerate{
  \item Train the binning model using \code{obwoe()} on the training set
  \item Save the fitted object with \code{saveRDS()}
  \item Load and apply using \code{obwoe_apply()} on new data
}

The WoE-transformed features can be used directly as inputs to logistic
regression or other linear models, enabling interpretable credit scorecards.
}
}
\examples{
\donttest{
# =============================================================================
# Example 1: Basic Usage - Train and Apply
# =============================================================================
set.seed(42)
n <- 1000

# Training data
train_df <- data.frame(
  age = rnorm(n, 40, 15),
  income = exp(rnorm(n, 10, 0.8)),
  education = sample(c("HS", "BA", "MA", "PhD"), n, replace = TRUE),
  target = rbinom(n, 1, 0.15)
)

# Fit binning model
model <- obwoe(train_df, target = "target")

# New data for scoring (could be validation/test set)
new_df <- data.frame(
  age = c(25, 45, 65),
  income = c(20000, 50000, 80000),
  education = c("HS", "MA", "PhD")
)

# Apply transformations
scored <- obwoe_apply(new_df, model)
print(scored)

# Use WoE features for downstream modeling
woe_cols <- grep("_woe$", names(scored), value = TRUE)
print(woe_cols)

# =============================================================================
# Example 2: Without Original Features
# =============================================================================

scored_compact <- obwoe_apply(new_df, model, keep_original = FALSE)
print(scored_compact)
}

}
\references{
Siddiqi, N. (2006). Credit Risk Scorecards: Developing and Implementing
Intelligent Credit Scoring. \emph{John Wiley & Sons}.
\doi{10.1002/9781119201731}
}
\seealso{
\code{\link{obwoe}} for fitting the binning model,
  \code{\link{summary.obwoe}} for model diagnostics.
}
