#' Simulation for operating-characteristics applying NAP-based priors
#'
#' @description
#' Runs Monte Carlo simulations of an \code{E} vs \code{C2} trial and performs
#' Bayesian analysis with a NAP-based prior constructed by \code{\link{NAP_prior}()}.
#' The routine supports both single external study setting and multiple external studies
#' settings as encoded in the provided \code{NAP_prior} object, and works with either a fixed mixture
#' weight (mNAP) or an elastic, data-adaptive weight (eNAP).
#'
#' @param NAP_prior An object returned by \code{\link{NAP_prior}()} that contains
#'   the prior specification and (for eNAP) any calibrated tuning parameters \code{a}, \code{b}.
#' @param theta_EC2 Numeric scalar. True log-hazard ratio for \code{E} vs \code{C2}
#'   used to generate the direct trial data.
#' @param n_EC2 Integer. Total sample size for the simulated \code{E} vs \code{C2} trial.
#' @param lambda Numeric scalar \eqn{> 0}. Randomization ratio \code{E:C2}; e.g.,
#'   \code{lambda = 2} means 2:1 allocation to \code{E}:\code{C2}.
#' @param sim_model Character string. Event-time model used to simulate individual
#'   times; one of \code{"Exponential"} or \code{"Weibull"}.
#' @param model_param Named numeric vector for the baseline hazard of the control arm.
#'   For \code{sim_model = "Exponential"}, use \code{c(rate = ...)}.
#'   For \code{sim_model = "Weibull"}, use \code{c(shape = ..., rate = ...)}.
#' @param iter Integer. Total MCMC iterations per chain for JAGS (default \code{2000}).
#' @param chains Integer. Number of MCMC chains (default \code{4}).
#' @param seed Integer. Random seed for the simulation replicates.
#' @param nsim Integer. Number of Monte Carlo replicates (default \code{100}).
#' @param jags_model Either a length-1 character string containing JAGS model code
#'   (e.g., a packaged object such as \code{jags_model_RE}) or a file path to a
#'   \code{.txt} JAGS model. If \code{NULL}, a default FE/RE model is chosen to
#'   match the \code{NAP_prior} mode.
#'
#' @return A data frame with one row per replicate containing:
#' \itemize{
#'   \item \code{post_mean}, \code{post_sd}, \code{low95}, \code{hi95}
#'         — posterior mean, SD, and 95\% interval for \code{theta_{E,C2}}.
#'   \item \code{prob_E_better} — posterior probability \code{theta_{E,C2} < 0}.
#'   \item \code{prior_weight}, \code{post_weight} — prior and updated weights
#'         used in the mixture (for eNAP, \code{prior_weight} is \code{w(Z)}).
#'   \item \code{sigma_hat} — posterior mean of between-study SD (RE only; \code{NA} for FE).
#' }
#'
#'
#' @export
#' @importFrom R2jags jags 

NAP_oc <- function(NAP_prior   = NULL,
                       theta_EC2   =  0.00,
                       n_EC2       =  200,
                       lambda      = 2,
                       sim_model   = c("Exponential","Weibull"),
                       model_param = 0.05,
                       iter        = 2000,
                       chains      = 4,
                       seed        = 123,
                       nsim        = 100,
                       jags_model  = NULL) {
  
  ## ---- choose model default (if not supplied)
  if (!inherits(NAP_prior, "NAP_prior"))
    stop("`NAP_prior` must be an object returned by NAP_prior().", call. = FALSE)
  
  if (!is.numeric(theta_EC2) || length(theta_EC2) != 1L || !is.finite(theta_EC2))
    stop("`theta_EC2` must be a finite scalar.", call. = FALSE)
  
  if (!is.numeric(n_EC2) || length(n_EC2) != 1L || !is.finite(n_EC2) || n_EC2 < 2)
    stop("`n_EC2` must be an integer >= 2.", call. = FALSE)
  
  if (!is.numeric(lambda) || length(lambda) != 1L || !is.finite(lambda) || lambda <= 0)
    stop("`lambda` must be a positive scalar.", call. = FALSE)
  
  sim_model <- match.arg(sim_model)
  
  # validate model_param shape
  if (is.null(model_param))
    stop("`model_param` must be provided: c(rate=...) for Exponential; c(shape=..., rate=...) for Weibull.", call. = FALSE)
  
  if (is.list(model_param)) model_param <- unlist(model_param, use.names = TRUE)
  if (is.null(names(model_param)))
    stop("`model_param` must be a *named* numeric vector.", call. = FALSE)
  
  if (sim_model == "Exponential") {
    if (is.null(model_param["rate"]) || !is.finite(model_param["rate"]) || model_param["rate"] <= 0)
      stop("For Exponential, `model_param` must include positive `rate`.", call. = FALSE)
  } else {
    need <- c("shape","rate")
    if (!all(need %in% names(model_param))) stop("For Weibull, provide model_param = c(shape=..., rate=...).", call. = FALSE)
    if (!is.finite(model_param["shape"]) || model_param["shape"] <= 0) stop("Weibull `shape` must be > 0.", call. = FALSE)
    if (!is.finite(model_param["rate"])  || model_param["rate"]  <= 0) stop("Weibull `rate` must be > 0.",  call. = FALSE)
  }
  
  # ---- Extract prior metadata (defines FE vs RE, weight mode, etc.)
  d          <- NAP_prior$details
  weight_med <- NAP_prior$weight_mode
  is_RE      <- identical(NAP_prior$mode, "Multiple external trials")
  
  # ---- choose model default (after we know is_RE)
  if (is.null(jags_model)) {
    jags_model <- .get_pkg_model(is_RE)   # <- returns the JAGS model TEXT
  }
  # pull inputs for indirect components
  y_C2C1_ori <- d$y_inputs$y_C2C1_original
  s_C2C1_ori <- d$s_inputs$s_C2C1_original
  y_C2C1_RE  <- d$y_inputs$y_C2C1_RE
  s_C2C1_RE  <- d$s_inputs$s_C2C1_RE
  y_EC1      <- d$y_inputs$y_EC1
  s_EC1      <- d$s_inputs$s_EC1
  

  model_path <- .model_path_from(jags_model)
  
  reps <- seq_len(nsim)
  set.seed(seed)
  out <- purrr::map_dfr(reps, function(i) {
    ans <- .one_rep(
      weight_med   = weight_med,
      is_RE        = is_RE,
      w_inform     = d$w_inform,
      a            = d$a,
      b            = d$b,
      y_C2C1_ori   = y_C2C1_ori,
      y_C2C1_RE    = y_C2C1_RE,
      s_C2C1_ori   = s_C2C1_ori,
      s_C2C1_RE    = s_C2C1_RE,
      y_EC1        = y_EC1,
      s_EC1        = s_EC1,
      theta_EC2    = theta_EC2,
      n_EC2        = n_EC2,
      lambda       = lambda,
      sim_model    = sim_model,
      model_param  = model_param,
      iter         = iter,
      chains       = chains,
      model_path   = model_path
    )
    data.frame(rep = i, t(ans), check.names = FALSE)
  })
  
  rownames(out) <- NULL
  out
}

# ==== helpers (not exported) ===============================================

#' @keywords internal
.one_rep <- function(weight_med,
                        is_RE,
                        w_inform, a, b,
                        y_C2C1_ori, y_C2C1_RE,
                        s_C2C1_ori, s_C2C1_RE,
                        y_EC1, s_EC1,
                        theta_EC2, n_EC2, lambda,
                        sim_model, model_param,
                        iter, chains, model_path) {
  
  # 1) simulate direct EC2 & summarize
  df_EC2  <- .sim_trial(n = n_EC2, lambda = lambda, theta = theta_EC2,
                        model = sim_model, param = model_param)
  sum_dir <- .tidy_cox(df_EC2)
  y_dir   <- sum_dir$diff
  se_dir  <- sum_dir$se
  
  # 2) prior weight
  if (identical(weight_med, "fixed")) {
    weight <- w_inform
  } else {
    y_C2C1 <- if (is_RE) y_C2C1_RE else y_C2C1_ori
    s_C2C1 <- if (is_RE) s_C2C1_RE else s_C2C1_ori  # variance
    n_eff<-1/(se_dir^2 + s_EC1 + s_C2C1)
    Z      <- n_eff^(-0.25)*abs(y_dir - (y_EC1 - y_C2C1)) / sqrt(se_dir^2 + s_EC1 + s_C2C1)
    weight <- 1 / (1 + exp(a + b * log(Z + 1)))
  }
  
  # 3) build JAGS data & run
  if (!is_RE) {
    data_mix <- list(
      y_ind  = c(y_EC1, y_C2C1_ori),
      se_ind = c(sqrt(s_EC1), sqrt(s_C2C1_ori)),
      y_dir  = y_dir, se_dir = se_dir,
      w      = weight
    )
    par_save <- c("theta_EC2","z")
  } else {
    data_mix <- list(
      y_dir = y_dir, se_dir = se_dir,
      y_EC1 = y_EC1, se_EC1 = sqrt(s_EC1),
      m_h   = length(y_C2C1_ori),
      y_ext = as.numeric(y_C2C1_ori),
      se_ext= as.numeric(sqrt(s_C2C1_ori)),
      w     = weight
    )
    par_save <- c("theta_EC2","z","sigma","d_C2C1")
  }
  
  fit <- R2jags::jags(
    data                = data_mix,
    parameters.to.save  = par_save,
    model.file          = model_path,
    n.chains            = chains,
    n.iter              = iter,
    n.burnin            = max(1000, floor(iter/2)),
    quiet               = TRUE,
    progress.bar        = "none"
  )
  
  post   <- fit$BUGSoutput$sims.list
  theta  <- post$theta_EC2
  z_mean <- mean(post$z)
  
  c(
    post_mean     = mean(theta),
    post_sd       = stats::sd(theta),
    low95         = stats::quantile(theta, 0.025),
    hi95          = stats::quantile(theta, 0.975),
    prob_E_better = mean(theta < 0),
    prior_weight  = weight,
    post_weight   = z_mean,
    sigma_hat     = if (!is.null(post$sigma)) mean(post$sigma) else NA_real_
  )
}
# --- shared internal helpers --------------------------------
#' @keywords internal
.model_path_from <- function(model) {
  if (is.character(model) && length(model) == 1L) {
    if (file.exists(model)) {
      return(normalizePath(model, winslash = "/", mustWork = TRUE))
    }
    out <- file.path(tempdir(), "model_RE_from_text.txt")
    writeLines(model, out, useBytes = TRUE)
    return(out)
  }
  stop("`model` must be either (i) a length-1 character string containing JAGS code ",
       "(e.g., your packaged `jags_model_RE` object), or (ii) a path to a .txt file.")
}
#' @keywords internal
.sim_trial <- function(n, lambda, theta,
                       model = c("Exponential", "Weibull"),
                       param) {
  model <- match.arg(model)
  stopifnot(is.numeric(n), length(n) == 1L, is.finite(n), n >= 2,
            is.numeric(lambda), length(lambda) == 1L, is.finite(lambda), lambda > 0,
            is.numeric(theta), length(theta) == 1L, is.finite(theta))
  
  # normalize/validate param
  if (is.list(param)) param <- unlist(param, use.names = TRUE)
  if (is.null(names(param))) stop("`param` must be named.")
  
  # allocate E:C2 = lambda:1 and shuffle arms
  n <- as.integer(n)
  n_E <- max(1L, min(n - 1L, round(lambda * n / (1 + lambda))))
  n_C <- n - n_E
  arm <- sample(c(rep("E", n_E), rep("C2", n_C)))
  
  time <- numeric(n)
  
  if (model == "Exponential") {
    if (is.null(param["rate"])) stop("For Exponential, provide param = c(rate = ...).")
    rate_C <- as.numeric(param["rate"]); if (!is.finite(rate_C) || rate_C <= 0) stop("rate > 0 required.")
    rate_E <- rate_C * exp(theta)  # HR = exp(theta)
    
    idxE <- which(arm == "E");  idxC <- which(arm == "C2")
    time[idxE] <- rexp(length(idxE), rate = rate_E)
    time[idxC] <- rexp(length(idxC), rate = rate_C)
    
  } else { # Weibull
    need <- c("shape","rate")
    if (!all(need %in% names(param))) stop("For Weibull, provide param = c(shape = ..., rate = ...).")
    k <- as.numeric(param["shape"]); rate_C <- as.numeric(param["rate"])
    if (!is.finite(k) || k <= 0) stop("shape > 0 required.")
    if (!is.finite(rate_C) || rate_C <= 0) stop("rate > 0 required.")
    
    # Base R rweibull(shape=k, scale=s); hazard = k*s^{-k}*t^{k-1}
    scale_C <- rate_C^(-1 / k)
    rate_E  <- rate_C * exp(theta)   # PH: HR = exp(theta)
    scale_E <- rate_E^(-1 / k)
    
    idxE <- which(arm == "E");  idxC <- which(arm == "C2")
    time[idxE] <- rweibull(length(idxE), shape = k, scale = scale_E)
    time[idxC] <- rweibull(length(idxC), shape = k, scale = scale_C)
  }
  
  data.frame(
    id     = seq_len(n),
    arm    = arm,
    time   = time,
    status = 1L,            # no censoring
    stringsAsFactors = FALSE
  )
}
#' @keywords internal
.tidy_cox <- function(df) {
  # Ensure the coef is log-HR for E vs C2
  df$arm <- factor(df$arm, levels = c("C2","E"))
  fit <- survival::coxph(survival::Surv(time, status) ~ arm, data = df)
  data.frame(
    study = "E vs C2",
    diff  = as.numeric(coef(fit)),                 # log-HR (E relative to C2)
    se    = sqrt(diag(stats::vcov(fit))),
    row.names = NULL,
    check.names = FALSE
  )
}
.get_pkg_model <- function(is_RE, pkg = "NAPrior") {
  ns <- asNamespace(pkg)
  nm <- if (is_RE) "jags_model_RE" else "jags_model_FE"
  txt <- get0(nm, envir = ns, inherits = FALSE)
  if (!is.character(txt) || length(txt) != 1L || !nzchar(txt)) {
    stop(sprintf("Internal model `%s` not found or not a length-1 character string.", nm), call. = FALSE)
  }
  txt
}