#' @title Simulates Data Sets Based on a Structural Equation Model (SEM).
#' @description
#' This function generates data sets based on a specified SEM.
#' The simulated data are organized such that the first column represents case identifiers, while the subsequent columns contain the simulated item responses.
#' For example, in a model with two factors and three items per factor, the column labels will follow the format: "ID, F1_x1, F1_x2, F1_x3, F2_x1, F2_x2, F2_x3".
#' The number of rows corresponds to the sample number of the data.
#' In addition to the generated data sets, two supplementary files are also saved:
#' (1) "Model_Info.dat" — containing the factor correlation and factor loading matrices
#' (2) "Data_List.dat" — listing the names of all generated data files.
#'
#' @author Fatih Orçan
#' @importFrom Matrix chol
#' @importFrom stats rnorm
#' @importFrom utils write.table
#' @param nd An integer, the number of data sets to be generated.
#' @param ss An integer, the sample size per data set (must be greater than 10).
#' @param fcors The factor correlation matrix, which must be symmetric. For one-factor models, this should be "matrix(1,1,1)".
#' @param loading The factor loading matrix. Columns correspond to factors, while non-zero rows specify the number of items associated with each factor.
#' @param f.loc File path indicating the directory where the generated data sets will be saved.
#' @export
#' @examples
#'
#' fc<-fcors.value(nf=3, cors=c(1,.5,.6,.5,1,.4,.6,.4,1))
#' fl<-loading.value(nf=3, fl.loads=c(.5,.5,.5,0,0,0,0,0,0,0,0,.6,.6,.6,0,0,0,0,0,0,0,0,.4,.4))
#'
#' sim.normal(nd=10, ss=1000, fcors=fc, loading<-fl,  f.loc=tempdir())
#'

sim.normal<-function(nd=10, ss=100, fcors, loading, f.loc){
  nd<-as.integer(nd)
  if (is.na(nd)==TRUE | nd < 1) {
    message("Error: The number of dataset should be a positive integer.")
    stop()}
  ss<-as.integer(ss)
  if (is.na(ss)==TRUE | ss < 10) {
    message("Error: The sample size should be an integer larger than 10.")
    stop()}

  if (all(is.numeric(fcors))!= TRUE | any(fcors>1) | any(fcors< -1)) {
    message("Error: Correlation values should be between -1 and +1.")
    stop()}

  nf<-dim(fcors)[1]

  if (all(is.numeric(loading))!= TRUE | any(loading>=1) | any(loading< 0)) {
    message("Error: Factor loadings should be smaller than 1 and larger than 0.")
    message("Note: The loading matrix should only not include cross loading.")
    stop()}

  if(dim(loading)[2]!=nf){
    message("Error: Number of factors entered is not matching with the earlier values. Please check your parameters.")
    stop()}

  U<-chol(fcors) #Cholesky decomposition
  f.names<-vector(mode = "character", length=nd)
  nfloads<-colSums(loading!=0)
  nitem<-dim(loading)[1]

  e.loading<-loading**2
  e.loading<-sqrt(1-apply(e.loading,1,sum))

  for(n.d in 1:nd){
    # Generation for Factor Scores
    f.scores<-matrix(NA,ss,nf)
    escores<-matrix(NA,ss,ncol=nitem)
    for(i in 1:nf){
      f.scores[,i]<-rnorm(ss,0,1)}
    for(i in 1:nitem){
      escores[,i]<-rnorm(ss,0,1)}
    cor.f.scores<-f.scores%*%U

    item.scores<-matrix(0,nrow = ss,ncol= nitem)
    for(j in 1:ss){
      item.scores[j,]<-loading%*%cor.f.scores[j,]+e.loading*escores[j,]
      f.names[n.d]<-c(paste("Data_",n.d,".dat", sep = ""))
    }

    item.scores<-cbind(c(1:ss),item.scores)
    message(paste(round(100*n.d/nd, 4), "% of simulation is completed...", sep = " "))
    write.table(item.scores, file=paste(f.loc,"/Data_",n.d,".dat", sep=""), sep = "\t", col.names = FALSE, row.names = FALSE,quote = FALSE)

  }

  write.table(f.names,file=paste(f.loc,"/Data_List.dat", sep = ""), col.names = FALSE, row.names = FALSE, quote = FALSE)
  message("Data generation has been completed...")

  sink(paste(f.loc,"/Model_Info.dat", sep = ""))
  print("The factor correlation matrix:")
  print(fcors)
  print("The factor loading matrix:")
  print(loading)
  sink()

}
