# Evaluation and comparison of stable distribution estimation methods
# =============================================================================
# ESTIMATION EVALUATION
# =============================================================================


#' Evaluate estimation method using MSE over multiple trials
#'
#' Computes the mean squared error (MSE) of a parameter estimation function over several trials.
#'
#' @param estimator_fn Function. Estimation function to evaluate.
#' @param true_params List. True parameters: alpha, beta, gamma, delta.
#' @param n Integer. Number of samples per trial.
#' @param trials Integer. Number of trials to run.
#' @param seed Integer. Random seed for reproducibility.
#' @return Numeric value representing the average MSE across trials.
#' @export
evaluate_estimation_method <- function(estimator_fn, true_params, n = 1000, trials = 20, seed = 42) {
  mse_list <- numeric(trials)
  for (i in 1:trials) {
    X <- rstable(n, true_params$alpha, true_params$beta,
                 gamma = true_params$gamma, delta = true_params$delta)
    est <- estimator_fn(X)
    mse <- mean(c(
      (est$alpha - true_params$alpha)^2,
      (est$beta  - true_params$beta)^2,
      (est$gamma - true_params$gamma)^2,
      (est$delta - true_params$delta)^2
    ))
    mse_list[i] <- mse
  }
  return(mean(mse_list))
}

# =============================================================================
# METHOD COMPARISON
# =============================================================================


#' Compare McCulloch, ECF, and MLE methods across parameter configurations
#'
#' Evaluates and compares three estimation methods (McCulloch, ECF, MLE) over a set of parameter configurations.
#'
#' @param parameter_configs Named list of parameter sets (each a list with alpha, beta, gamma, delta).
#' @param trials Integer. Number of trials per configuration.
#' @param n Integer. Number of samples per trial.
#'
#' @return A named list of results with MSE values for each method and configuration.
#' @export
compare_methods_across_configs <- function(parameter_configs, trials = 20, n = 1000) {
  results <- list()
  # Generate lookup table and interpolators
  alpha_grid <- seq(0.6, 2.0, length.out = 15)
  beta_grid  <- seq(-1.0, 1.0, length.out = 15)
  table <- generate_mcculloch_table(alpha_grid, beta_grid)
  interpolators <- build_mcculloch_interpolators(table)

  for (name in names(parameter_configs)) {
    message("Running:", name, "\n")
    true_params <- parameter_configs[[name]]

    # McCulloch - FIXED: Use correct function signature
    mc_mse <- evaluate_estimation_method(
      function(X) mcculloch_lookup_estimate(X, interpolators = interpolators),
      true_params, n = n, trials = trials
    )

    # ECF
    ecf_mse <- evaluate_estimation_method(
      function(X) ecf_estimate_all(X),
      true_params, n = n, trials = trials
    )

    # MLE
    mle_mse <- evaluate_estimation_method(
      function(X) mle_estimate(X),
      true_params, n = n, trials = trials
    )

    results[[name]] <- list(
      McCulloch = mc_mse,
      ECF       = ecf_mse,
      MLE       = mle_mse
    )
  }
  return(results)
}

# =============================================================================
# EM VS EM WITH GIBBS COMPARISON
# =============================================================================


#' Compare standard EM and EM with Gibbs sampling using kernel ECF
#'
#' Compares log-likelihood and weight estimates between standard EM and EM with Gibbs sampling.
#'
#' @param data Numeric vector. Observed data.
#' @param n_runs Integer. Number of runs for comparison.
#'
#' @return Data frame with log-likelihoods and weights for each method across runs.
#' @importFrom grDevices png dev.off rgb
#' @export
compare_em_vs_em_gibbs <- function(data, n_runs = 20) {
  results <- data.frame()
  for (seed in 1:n_runs) {
    # Standard EM
    em_result <- em_fit_alpha_stable_mixture(data)
    params_em1 <- em_result$params1
    params_em2 <- em_result$params2
    w_em <- em_result$weight
    pdf_em <- w_em * r_stable_pdf(data, params_em1[1], params_em1[2], params_em1[3], params_em1[4]) +
      (1 - w_em) * r_stable_pdf(data, params_em2[1], params_em2[2], params_em2[3], params_em2[4])
    loglik_em <- sum(log(pmax(pdf_em, 1e-300)))
    # EM with Gibbs (kernel ECF)
    gibbs_result <- em_estimate_stable_kernel_ecf_with_gibbs(data, n_runs, tol = 1e-4)
    params_gibbs1 <- gibbs_result$params1
    params_gibbs2 <- gibbs_result$params2
    w_gibbs <- gibbs_result$weight
    pdf_gibbs <- w_gibbs * r_stable_pdf(data, params_gibbs1[1], params_gibbs1[2], params_gibbs1[3], params_gibbs1[4]) +
      (1 - w_gibbs) * r_stable_pdf(data, params_gibbs2[1], params_gibbs2[2], params_gibbs2[3], params_gibbs2[4])
    loglik_gibbs <- sum(log(pmax(pdf_gibbs, 1e-300)))
    results <- rbind(results, data.frame(
      seed        = seed,
      loglik_em   = loglik_em,
      loglik_gibbs= loglik_gibbs,
      w_em        = w_em,
      w_gibbs     = w_gibbs
    ))
  }
  return(results)
}
