% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/multimixMN.R
\name{multimixMN}
\alias{multimixMN}
\title{Multilayer MGM with bootstrap, intra/interlayer metrics, and quantile regions}
\usage{
multimixMN(
  data,
  layers,
  layer_rules,
  scale = TRUE,
  reps = 100,
  lambdaSel = c("CV", "EBIC"),
  lambdaFolds = 5,
  lambdaGam = 0.25,
  alphaSeq = 1,
  alphaSel = "CV",
  alphaFolds = 5,
  alphaGam = 0.25,
  k = 2,
  ruleReg = "AND",
  threshold = "LW",
  overparameterize = FALSE,
  thresholdCat = TRUE,
  quantile_level = 0.95,
  covariates = NULL,
  exclude_from_cluster = NULL,
  seed_model = NULL,
  seed_boot = NULL,
  treat_singletons_as_excluded = FALSE,
  cluster_method = c("louvain", "fast_greedy", "infomap", "walktrap", "edge_betweenness"),
  compute_loadings = TRUE,
  boot_what = c("general_index", "interlayer_index", "bridge_index", "excluded_index",
    "community", "loadings"),
  save_data = FALSE,
  progress = TRUE
)
}
\arguments{
\item{data}{A \code{data.frame} (n x p) with variables in columns.
Variables may be numeric, integer, logical, or factors.
Character and Date/POSIXt variables are not supported and must be converted
prior to model fitting.
Variable types are internally mapped to MGM types as follows:
continuous numeric (double) variables are treated as Gaussian;
integer variables are treated as Poisson unless they take only values
in \{0,1\}, in which case they are treated as binary categorical;
factors and logical variables are treated as categorical.
Binary categorical variables (two-level
factors and logical variables) are internally recoded to \{0,1\} for model
fitting. The original input data are not modified.}

\item{layers}{A named vector (names = variable names) assigning each node to a
layer (character or factor). Must cover all columns of \code{data}
except variables listed in \code{covariates} (treated as adjustment covariates).}

\item{layer_rules}{A square matrix (L × L), where L is the number of layers.
Row and column names must match the layer names. Entries equal to
\code{TRUE} or \code{1} allow cross-layer edges between the corresponding
pair of layers, while \code{FALSE} or \code{0} disallow them.
The matrix is symmetrized internally. Diagonal entries are ignored
(intralayer edges are always permitted).}

\item{scale}{Logical; if \code{TRUE} (default) Gaussian variables
(\code{type == "g"}) are z-standardized internally by \code{mgm()}. Use
\code{scale = FALSE} if your data are already standardized.}

\item{reps}{Integer (>= 0). Number of bootstrap replications (row resampling
with replacement). If \code{reps = 0}, no bootstrap is performed.}

\item{lambdaSel}{Method for lambda selection in \code{mgm}:
\code{"CV"} or \code{"EBIC"}.}

\item{lambdaFolds}{Number of folds for CV (if \code{lambdaSel = "CV"}).}

\item{lambdaGam}{EBIC gamma parameter (if \code{lambdaSel = "EBIC"}).}

\item{alphaSeq}{Alpha parameters of the elastic net penalty (values between 0 and 1).}

\item{alphaSel}{Method for selecting the alpha parameter:
\code{"CV"} or \code{"EBIC"}.}

\item{alphaFolds}{Number of folds for CV (if \code{alphaSel = "CV"}).}

\item{alphaGam}{EBIC gamma parameter (if \code{alphaSel = "EBIC"}).}

\item{k}{Integer (>= 1). Order of modeled interactions.}

\item{ruleReg}{Rule to combine neighborhood estimates:
\code{"AND"} or \code{"OR"}.}

\item{threshold}{Threshold below which edge-weights are set to zero:
Available options are \code{"LW"}, \code{"HW"}, or \code{"none"}.
\code{"LW"} applies the threshold proposed by Loh & Wainwright;
\code{"HW"} applies the threshold proposed by Haslbeck & Waldorp;
\code{"none"} disables thresholding. Defaults to \code{"LW"}.}

\item{overparameterize}{Logical; controls how categorical interactions are
parameterized in the neighborhood regressions. If \code{TRUE}, categorical
interactions are represented using a fully over-parameterized design matrix
(i.e., all category combinations are explicitly modeled). If \code{FALSE},
the standard \code{glmnet} parameterization is used, where one category
serves as reference. For continuous variables, both parameterizations are
equivalent. The default is \code{FALSE}. The over-parameterized option may
be advantageous when distinguishing pairwise from higher-order interactions.}

\item{thresholdCat}{Logical; if \code{FALSE} thresholds of categorical
variables are set to zero.}

\item{quantile_level}{Level of the central bootstrap quantile region (default \code{0.95}).
Must be a single number between 0 and 1.}

\item{covariates}{Character vector. Variables used as adjustment covariates
in model estimation.}

\item{exclude_from_cluster}{Character vector of node names. Nodes in this set
are excluded from community detection in addition to \code{covariates}.}

\item{seed_model}{Optional integer seed for reproducibility of the initial
MGM fit.}

\item{seed_boot}{Optional integer seed passed to \code{future.apply} for
reproducibility of bootstrap replications.}

\item{treat_singletons_as_excluded}{Logical; if \code{TRUE}, singleton
communities (size 1) are treated as "excluded" when computing bridge
metrics and related summaries.}

\item{cluster_method}{Community detection algorithm applied within each
layer. One of \code{"louvain"}, \code{"fast_greedy"}, \code{"infomap"},
\code{"walktrap"}, or \code{"edge_betweenness"}.}

\item{compute_loadings}{Logical; if \code{TRUE} (default), compute community loadings
(\code{EGAnet::net.loads}). Only supported for Gaussian, Poisson, and binary
categorical nodes; otherwise loadings are skipped and the reason is
stored in \code{community_loadings$reason}.}

\item{boot_what}{Character vector specifying which quantities to bootstrap.
Valid options are:
\code{"general_index"} (intralayer centrality indices),
\code{"interlayer_index"} (interlayer-only node metrics),
\code{"bridge_index"} (bridge metrics for nodes in communities),
\code{"excluded_index"} (bridge metrics for nodes treated as excluded),
\code{"community"} (community memberships),
\code{"loadings"} (within-layer community loadings, only if
\code{compute_loadings = TRUE}),
and \code{"none"} (skip all node-level bootstrap: only edge-weight
bootstrap is performed if \code{reps > 0}).}

\item{save_data}{Logical; if \code{TRUE}, store the original data in the output object.}

\item{progress}{Logical; if \code{TRUE} (default), show a bootstrap progress bar.}
}
\value{
An object of class \code{c("mixmashnet", "multimixMN_fit")}. The returned
list contains at least the following components:
\describe{
  \item{\code{call}}{
  The matched function call.
  }
  \item{\code{settings}}{
  List of main settings used in the call, including
   \code{reps}, \code{cluster_method}, \code{covariates},
    \code{exclude_from_cluster}, \code{treat_singletons_as_excluded},
    \code{boot_what}).
  }
 \item{\code{data_info}}{
    List with information derived from the input data used for model setup:
    \code{mgm_type_level} (data frame with one row per variable, reporting
    the original R class and the inferred MGM \code{type} and \code{level},
    as used in the call to \code{mgm::mgm}),
    and \code{binary_recode_map} (named list describing the mapping from
    original binary labels to the internal \{0,1\} coding used for model fitting).
  }
  \item{\code{model}}{
    List with:
    \code{mgm} (the fitted \code{mgm} object),
    \code{nodes} (character vector of all node names),
    \code{n} (number of observations),
    \code{p} (number of variables), and
    \code{data} (if \code{save_data = TRUE}))
  }
  \item{\code{layers}}{
   List describing the multilayer structure
    (assignment of nodes to layers, \code{layer_rules} matrix used and color of each layer in \code{palette}).
  }
  \item{\code{layer_fits}}{
   Named list (one element per layer) with
    single layer fits, including community structure, node-level statistics,
    edge-level statistics, bridge metrics, and (optionally) community loadings
    with bootstrap information.
  }
  \item{\code{interlayer}}{
  List collecting interlayer-only node metrics
  (strength, expected influence, closeness, betweenness, with or without
  bootstrap) and cross-layer edge summaries for each allowed pair of
  layers.
  }
  \item{\code{graph}}{
  List containing a global \pkg{igraph} object built on
  the retained nodes (\code{keep_nodes_graph}), with vertex attributes
  such as \code{name}, \code{layer}, \code{membership}, and edge attributes
  such as \code{weight}, \code{abs_weight}, \code{sign},
  \code{type} (intra vs inter) and \code{layer_pair}.
  }
}
}
\description{
Estimates a multilayer Mixed Graphical Model (MGM) using the estimation
framework implemented in the \pkg{mgm} package, with a masking scheme that
enforces which cross-layer edges are allowed according to \code{layer_rules}.
Within each layer, the function computes community structure and performs
non-parametric row-bootstrap to obtain node centrality indices, edge weights,
and bridge metrics, including metrics for nodes treated as excluded. Optionally,
within-layer community loadings can also be estimated and bootstrapped.
The function additionally returns interlayer-only node metrics and summaries
of cross-layer edge weights.
}
\details{
This function does \strong{not} call \code{future::plan()}. To enable
parallel bootstrap, set a plan (e.g. \code{future::plan(multisession)}) before
calling \code{multimixMN()}. If \code{"none"} is the only element of
\code{boot_what} and \code{reps > 0}, node-level metrics are not
bootstrapped, but intra and interlayer edge-weight bootstrap and the
corresponding quantile regions are still computed.
}
\examples{

\donttest{
data(nhanes)

bio_vars  <- c("ALT", "AST", "HDL", "HbA1c")
ant_vars  <- c("BMI", "Waist", "ArmCirc", "LegLength")
life_vars <- c("Smoke", "PhysicalActivity", "Drug")
covs      <- c("Age", "Gender", "MonInc")

df <- nhanes[, c(bio_vars, ant_vars, life_vars, covs)]

# Layer assignment (must cover all columns except covariates)
layers <- c(
  setNames(rep("bio",  length(bio_vars)),  bio_vars),
  setNames(rep("ant",  length(ant_vars)),  ant_vars),
  setNames(rep("life", length(life_vars)), life_vars)
)

# Allow cross-layer edges bio<->ant and ant<->life; disallow bio<->life
layer_rules <- matrix(0, nrow = 3, ncol = 3,
                      dimnames = list(c("bio","ant","life"),
                                      c("bio","ant","life")))
layer_rules["bio","ant"]  <- 1
layer_rules["ant","life"] <- 1

fitM <- multimixMN(
  data = df,
  layers = layers,
  layer_rules = layer_rules,
  covariates = covs,
  lambdaSel = "EBIC",
  lambdaGam = 0.25,
  reps = 5,
  seed_model = 42,
  seed_boot = 42,
  compute_loadings = FALSE,
  progress = FALSE
)
fitM

# Plot the estimated network
set.seed(1)
plot(fitM, color_by = "layer")
}

}
\references{
Haslbeck, J. M. B., & Waldorp, L. J. (2020).
mgm: Estimating Time-Varying Mixed Graphical Models in High-Dimensional Data.
\emph{Journal of Statistical Software}, 93(8).
\doi{10.18637/jss.v093.i08}
}
