#' @title Maximum likelihood estimation under parametric survival function
#' @description \code{survival.mle} provides the maximum likelihood estimator and their variance-covariance matrix under parametric survival functions. This function also provides AIC and Kolmogorov-Smirnov distance to evaluate the model fitting.
#'
#' @usage survival.mle(
#' t.event,
#' event,
#' distribution = c("exponential", "weibull", "gamma", "log-normal", "burr3"),
#' par = c(0,0))
#'
#' @param t.event a vector for time-to-event.
#' @param event a vector for event indicator.
#' @param distribution a parametric distribution for survival function. Available options include; "exponential", "weibull", "gamma", "log-normal", "burr3".
#' @param par initial value(s) for the distribution parameters.
#'
#' @return estimate: a vector of MLE.
#' @return var: variance-covariance matrix of MLE.
#' @return AIC: Akaike information criteria.
#' @return KS: Kolmogorov-Smirnov distance between MLE and KM estimator.
#'
#' @examples
#' #MLE under exponential distribution
#' #set distribution parameter
#' lambda = 1
#'
#' #generate time to event
#' u = runif(100)
#' t.event = -log(u)/lambda
#' t.event = sort(t.event)
#'
#' #censoring indicator
#' tc = runif(100, 0, 1.5)
#' t.event = (tc >= t.event) * t.event + (tc < t.event) * tc
#' event = 1 * (tc > t.event)
#'
#' survival.mle(t.event, event, distribution = "exponential")
#'
#'
#'
#' @importFrom survival survfit
#' @importFrom survival Surv
#' @importFrom stats nlm
#' @export
#'

survival.mle <- function(t.event, event, distribution = c("exponential", "weibull", "gamma", "log-normal", "burr3"), par = c(0,0)) {

  KM = survfit(Surv(t.event, event)~1)

  switch (distribution,
    "exponential" = {
      lambda = sum(event == 1)/sum(t.event)
      V = lambda^2/sum(event == 1)
      aic = -2*(log(lambda)*sum(event == 1)-lambda*sum(t.event))+2
      ks = max(abs(summary(KM, times = t.event)$surv - exp(-lambda*t.event)))
      param.res = matrix(c(lambda), ncol = 1)
      colnames(param.res) = c("lambda")
      res = list(estimate = c(param.res), var = V, AIC = aic, KS = ks)
    },
    "weibull" = {
      ll <- function(x){
        -sum(event == 1)*x[1]-sum(event == 1)*x[2]-(exp(x[2])-1)*sum(log(t.event[event == 1]))+exp(x[1])*sum(t.event^exp(x[2]))
      }
      result_ml <- nlm(ll, c(par[1], par[2]), gradtol = 1e-7, hessian = TRUE)
      k = exp(result_ml$estimate[2])
      lambda = exp(result_ml$estimate[1])
      Hessian = matrix(c(result_ml$hessian[1,1]/exp(result_ml$estimate[1])^2,
                          result_ml$hessian[2,1]/exp(result_ml$estimate[1])/exp(result_ml$estimate[2]),
                          result_ml$hessian[2,1]/exp(result_ml$estimate[1])/exp(result_ml$estimate[2]),
                          result_ml$hessian[2,2]/exp(result_ml$estimate[2])^2), 2, 2)
      V = solve(Hessian, tol = 0L)
      aic <- 2*result_ml$minimum + 2*2
      ks <- max(abs(summary(KM, times = t.event)$surv - exp(-lambda*(t.event^k))))
      param.res = matrix(c(lambda, k), ncol = 2)
      colnames(param.res) = c("lambda", "k")
      res = list(estimate = param.res, var = V, AIC = aic, KS = ks)
    },
    "gamma" = {
      ll <- function(x) {
        -exp(x[1])*x[2]*sum(event == 1)-(exp(x[1])-1)*sum(log(t.event[event == 1]))+exp(x[2])*sum(t.event[event == 1])+log(gamma(exp(x[1])))*sum(event == 1)-sum(log(1-pgamma(t.event[event != 1], shape = exp(x[1]), scale = 1/exp(x[2]))))
      }
      result_ml <- nlm(ll, c(par[1], par[2]), hessian = TRUE, gradtol = 1e-8)
      beta = exp(result_ml$estimate[2])
      alpha = exp(result_ml$estimate[1])
      Hessian = matrix(c(result_ml$hessian[1,1]/exp(result_ml$estimate[1])^2,
                         result_ml$hessian[2,1]/exp(result_ml$estimate[1])/exp(result_ml$estimate[2]),
                         result_ml$hessian[2,1]/exp(result_ml$estimate[1])/exp(result_ml$estimate[2]),
                         result_ml$hessian[2,2]/exp(result_ml$estimate[2])^2), 2, 2)
      V <- solve(Hessian, tol = 0L)
      aic <- 2*result_ml$minimum + 2*2
      ks <- max(abs(summary(KM, times = t.event)$surv - (1-pgamma(t.event, shape = alpha, scale = 1/beta))))
      param.res = matrix(c(alpha, beta), ncol = 2)
      colnames(param.res) = c("alpha", "beta")
      res = list(estimate = param.res, var = V, AIC = aic, KS = ks)
    },
    "log-normal" = {
      ll <- function(x) {
        -sum(log(dlnorm(t.event[event == 1], meanlog = x[1], sdlog = exp(x[2]))))-sum(log(1-plnorm(t.event[event != 1], meanlog = x[1], sdlog = exp(x[2]))))
      }
      result_ml <- nlm(ll, c(par[1], par[2]), hessian = TRUE, gradtol = 1e-8)
      lsd = exp(result_ml$estimate[2])
      lmu = result_ml$estimate[1]
      Hessian = matrix(c(result_ml$hessian[1,1],
                         result_ml$hessian[2,1]/exp(result_ml$estimate[2]),
                         result_ml$hessian[2,1]/exp(result_ml$estimate[2]),
                         result_ml$hessian[2,2]/exp(result_ml$estimate[2])^2), 2, 2)
      V <- solve(Hessian, tol = 0L)
      aic <- 2*result_ml$minimum + 2*2
      ks <- max(abs(summary(KM, times = c(t.event))$surv - (1-plnorm(t.event, meanlog = lmu, sdlog = lsd))))
      param.res = matrix(c(lmu, lsd), ncol = 2)
      colnames(param.res) = c("mean", "sd")
      res = list(estimate = param.res, var = V, AIC = aic, KS = ks)
    },
    "burr3" = {
      ll <- function(x){
        -sum(event == 1)*x[1]-sum(event == 1)*x[2]+(exp(x[1])+1)*sum(log(t.event[event == 1]))+(exp(x[2])+1)*sum(log(1+t.event[event == 1]^(-exp(x[1]))))-sum(log(1-(1+t.event[event != 1]^(-exp(x[1])))^(-exp(x[2]))))
      }
      result_ml <- nlm(ll, c(par[1], par[2]), hessian = TRUE, gradtol = 1e-9)
      c = exp(result_ml$estimate[1])
      k = exp(result_ml$estimate[2])

      Hessian <- matrix(c(result_ml$hessian[1,1]/exp(result_ml$estimate[1])^2,
                          result_ml$hessian[2,1]/exp(result_ml$estimate[1])/exp(result_ml$estimate[2]),
                          result_ml$hessian[2,1]/exp(result_ml$estimate[1])/exp(result_ml$estimate[2]),
                          result_ml$hessian[2,2]/exp(result_ml$estimate[2])^2), 2, 2)
      V <- solve(Hessian, tol = 0L)

      aic <- 2*result_ml$minimum + 2*2
      ks <- max(abs(summary(KM, times = c(t.event))$surv - (1-(1+t.event^(-c))^(-k))))
      param.res = matrix(c(c, k), ncol = 2)
      colnames(param.res) = c("c", "k")
      res = list(estimate = param.res, var = V, AIC = aic, KS = ks)
    }
  )

  return(res)
}


#' @title Parametric calculation for the Mann-Whitney effect under survival copula models
#' @description \code{MW.comp} provides a parametric estimator for the Mann-Whitney effect under the parametric survival functions and copulas. See Nakazono, et al.(2024) for details.
#'
#' @usage MW.comp(
#' copula = c("clayton", "gumbel", "frank", "fgm", "gb"),
#' copula.param = 1,
#' s1 = c("exponential", "weibull", "gamma", "log-normal", "burr3"),
#' S1.param,
#' s2 = c("exponential", "weibull", "gamma", "log-normal", "burr3"),
#' S2.param
#' )
#'
#' @param copula copula family. Available options include; "clayton", "gumbel", "frank", "fgm", "gb".
#' @param copula.param the copula parameter.
#' @param s1 a parametric survival function for S1. Available options include; "exponential", "weibull", "gamma", "log-normal", "burr3".
#' @param S1.param the distribution parameter for S1.
#' @param s2 a parametric survival function for S2. Available options include; "exponential", "weibull", "gamma", "log-normal", "burr3".
#' @param S2.param the distribution parameter for S2.
#'
#' @return estimate of the Mann-Whitney effect.
#'
#' @details
#' copula.param is restricted as below: \cr
#' * "clayton"; copula.param >= 0
#' * "gumbel"; copula.param >= 0
#' * "frank"; -Inf < copula.param < Inf
#' * "fgm"; -1 =< copula.param =< 1
#' * "gb"; -1 =< copula.param =< 1
#'
#'
#' @examples
#' # Under the exponential survival functions and Clayton copula
#' MW.comp(
#'   copula = "clayton",
#'   copula.param = 1,
#'   s1 = "exponential",
#'   S1.param = 1,
#'   s2 = "exponential",
#'   S2.param = 2
#' )
#'
#'
#' @references
#' Nakazono, K., Lin, Y. C., Liao, G. Y., Uozumi, R., & Emura, T. (2024). Computation of the Mann–Whitney effect under parametric survival copula models. Mathematics, 12(10), 1453.
#'
#'
#' @importFrom stats integrate
#' @export
#'


MW.comp <- function(copula = c("clayton", "gumbel", "frank", "fgm", "gb"), copula.param = 1, s1 = c("exponential", "weibull", "gamma", "log-normal", "burr3"), S1.param, s2 = c("exponential", "weibull", "gamma", "log-normal", "burr3"), S2.param) {

  if (s1 == "exponential"){
    S1 <- function(x) {exp(-S1.param[1]*x)}
  }
  else if (s1 == "weibull"){
    S1 <- function(x){exp(-S1.param[1]*x^S1.param[2])}
  }
  else if (s1 == "gamma"){
    S1 <- function(x){1-pgamma(x, shape = S1.param[1], scale = 1/S1.param[2])}
  }
  else if (s1 == "log-normal"){
    S1 <- function(x){1-plnorm(x, meanlog = S1.param[1], sdlog = S1.param[2])}
  }
  else if (s1 == "burr3"){
    S1 <- function(x){1-(1+x^(-S1.param[1]))^(-S1.param[2])}
  }

  if (s2 == "exponential"){
    S2i <- function(x) {-log(x)/S2.param[1]}
  }
  else if (s2 == "weibull"){
    S2i <- function(x){(-log(x)/S2.param[1])^(1/S2.param[2])}
  }
  else if (s2 == "gamma"){
    S2i <- function(x){qgamma(1-x, shape = S2.param[1], scale = 1/S2.param[2])}
  }
  else if (s2 == "log-normal"){
    S2i <- function(x){qlnorm(1-x, meanlog = S2.param[1], sdlog = S2.param[2])}
  }
  else if (s2 == "burr3"){
    S2i <- function(x){((1-x)^(-1/S2.param[2])-1)^(-1/S2.param[1])}
  }

  if(copula == "clayton"){
    H_function <- function(x) {
      E = S1(S2i(x))
      #RR = (E == rep(1, length(E)))
      #E[RR] = 1-10^8
      B = E^(-copula.param) + max(x, 0.00001)^(-copula.param) - 1
      if (copula.param == 0) {
        res = E
      }
      else{
        res = max(x, 0.00001)^(-copula.param - 1) * B^(-1/copula.param - 1)
      }
    }
  }
  else if (copula == "gumbel"){
    H_function <- function(x) {
      E = S1(S2i(x))
      RR = (E == rep(1, length(E)))
      E[RR] = 1-10^8
      B = (-log(E))^(copula.param+1) + (-log(x))^(copula.param+1)
      res = exp(-(B^(1/(copula.param+1)))) * B^(-copula.param/(copula.param+1)) * ((-log(x))^copula.param)/x
      #ifelse(is.nan(res),0,res)
    }
  }
  else if (copula == "frank"){
    H_function <- function(x) {
      E = S1(S2i(x))
      B = exp(-copula.param*x)*(exp(-copula.param*E)-1)
      C = (exp(-copula.param)-1)+(exp(-copula.param*E)-1)*(exp(-copula.param*x)-1)
      res = B/C
      ifelse(is.nan(res),0,res)
    }
  }
  else if (copula == "fgm"){
    H_function <- function(x) {
      E = S1(S2i(x))
      B=E
      res = B+copula.param*B*(1-B)*(1-2*x)
      #ifelse(is.nan(res),0,res)
    }
  }
  else if (copula == "gb"){
    H_function <- function(x){
      E = S1(S2i(x))
      res = E*(1-copula.param*log(E))*(x^(-copula.param*log(E)))
      ifelse(is.nan(res),0,res)
    }
  }
  else if (copula == "indep"){
    H_function <- function(x){
      E = S1(S2i(x))
      E
    }
  }
  p <- NULL
  p <- integrate(H_function, 0, 1, abs.tol=1e-20)$value
  return(p)
}


#' @title Parametric calculation for the Mann-Whitney effect under the parametric copulas
#' @description \code{MW.Copula} provides a parametric estimator and confidence interval for the Mann-Whitney effect under the parametric survival functions and copulas. The result of this function includes results for logit-transformed estimators. See Nakazono, et al. (2025) for details.
#'
#' @usage MW.Copula(
#' t.event, event, group,
#' copula = c("clayton", "gumbel", "frank", "fgm", "gb"),
#' copula.param = 1,
#' s1 = c("exponential", "weibull", "gamma", "log-normal", "burr3"),
#' s2 = c("exponential", "weibull", "gamma", "log-normal", "burr3"),
#' par1 = c(0, 0),
#' par2 = c(0, 0),
#' alpha = 0.05,
#' logit = FALSE
#' )
#'
#'
#' @param t.event a vector of time-to-event data.
#' @param event a vector for event indicator.
#' @param group a vector for group indicator.
#' @param copula copula family.
#' @param copula.param the copula parameter.
#' @param s1 a parametric survival function for S1. Available options include; "exponential", "weibull", "gamma", "log-normal", "burr3".
#' @param s2 a parametric survival function for S2. Available options include; "exponential", "weibull", "gamma", "log-normal", "burr3".
#' @param par1 initial value(s) for the parameters of S1.
#' @param par2 initial value(s) for the parameters of S2.
#' @param alpha significance level.
#' @param logit logical; if TRUE, the estimator and CI is logit-transformed.
#'
#' @returns parameter: a vector of MLE.
#' @returns estimate: the Mann-Whitney effect estimator
#' @returns SE: the standard error
#' @returns CI: the 1-alpha% confidence interval for Mann-Whitney effect
#' @returns P.value: the P-value for testing the null hypothesis H0: p=1/2.
#' @returns kendall: Kendall's tau
#' @returns logit: the estimator and CI are logit-transformed or not.
#'
#' @examples
#' ##Mann-Whitney effect under exponential distributions
#' #set distribution parameter
#' lambda1 = 1
#' lambda2 = 2
#'
#' #generate time to event
#' u = runif(100)
#' t.event1 = -log(u) / lambda1
#' t.event1 = sort(t.event1)
#'
#' v = runif(100)
#' t.event2 = -log(v) / lambda2
#' t.event2 = sort(t.event2)
#'
#' #censoring indicator
#' t1c = runif(100, 0, 1.5)
#' t.event1 = (t1c >= t.event1) * t.event1 + (t1c < t.event1) * t1c
#' event1 = 1 * (t1c > t.event1)
#'
#' t2c = runif(100, 0, 0.8)
#' t.event2 = (t2c >= t.event2) * t.event2 + (t2c < t.event2) * t2c
#' event2 = 1 * (t2c > t.event2)
#'
#' t.event = c(t.event1, t.event2)
#' event = c(event1, event2)
#'
#' #group indicator
#' group = rep(c(1, 0), each = 100)
#'
#' MW.Copula(t.event, event, group,
#'           copula = "clayton", copula.param = 1,
#'           s1 = "exponential", s2 = "exponential", par1 = c(0, 0), par2 = c(0, 0), logit = FALSE)
#'
#'
#' @references
#' Nakazono, K., Uozumi, R., & Emura, T. (2025). Parametric inference for the Mann-Whitney effect under survival copula models, Statistical Papers, in press.
#'
#' @importFrom stats integrate dlnorm pgamma plnorm qgamma qlnorm qnorm pnorm
#' @export
#'


MW.Copula <- function(t.event, event, group,
                       copula = c("clayton", "gumbel", "frank", "fgm", "gb"), copula.param = 1,
                       s1 = c("exponential", "weibull", "gamma", "log-normal", "burr3"), s2 = c("exponential", "weibull", "gamma", "log-normal", "burr3"), par1 = c(0, 0), par2 = c(0, 0), alpha = 0.05, logit = FALSE) {

  t.event1 = t.event[group == 1]
  event1 = event[group == 1]
  t.event2 = t.event[group != 1]
  event2 = event[group != 1]

  res1 <- survival.mle(t.event = t.event1, event = event1, distribution = s1, par = par1)
  res2 <- survival.mle(t.event = t.event2, event = event2, distribution = s2, par = par2)

  switch (s1,
    "exponential" = {
      p11 = res1$estimate[1]
      p12 = 0
      V11 = res1$var[1]
      V12 = 0
      CV1 = 0
      S1 <- function(x) {exp(-p11*x)}
      dS1.p1 <- function(x) {-x*exp(-p11*x)}
      dS1.p2 <- function(x) {0}
    },
    "weibull" = {
      p11 = res1$estimate[1]
      p12 = res1$estimate[2]
      V11 = res1$var[1,1]
      V12 = res1$var[2,2]
      CV1 = res1$var[1,2]
      S1 <- function(x) {exp(-p11*x^p12)}
      dS1.p1 <- function(x) {-x^p12*exp(-p11*x^p12)}
      dS1.p2 <- function(x) {-p12*x^p12*log(x)*exp(-p11*x^p12)}
    },
    "gamma" = {
      p11 = res1$estimate[1]
      p12 = res1$estimate[2]
      V11 = res1$var[1,1]
      V12 = res1$var[2,2]
      CV1 = res1$var[1,2]
      S1 <- function(x) {1-pgamma(x, shape = p11, scale = 1/p12)}
      dS1.p1 <- function(x) {
        h = 1e-5
        ((1-pgamma(x, shape = p11+h, rate = p12)) - (1-pgamma(x, shape = p11, rate = p12))) / h
      }
      dS1.p2 <- function(x) {-(p12*x)^(p11-1)*exp(-p12*x)*x/gamma(p11)}
    },
    "log-normal" = {
      p11 = res1$estimate[1]
      p12 = res1$estimate[2]
      V11 = res1$var[1,1]
      V12 = res1$var[2,2]
      CV1 = res1$var[1,2]
      S1 <- function(x) {1-plnorm(x, meanlog = p11, sdlog = p12)}
      dS1.p1 <- function(x) {
        h = 1e-5
        ((1-plnorm(x, meanlog = p11+h, sdlog = p12)) - (1-plnorm(x, meanlog = p11, sdlog = p12))) / h
      }
      dS1.p2 <- function(x) {
        h = 1e-5
        ((1-plnorm(x, meanlog = p11, sdlog = p12+h)) - (1-plnorm(x, meanlog = p11, sdlog = p12))) / h
      }
    },
    "burr3" = {
      p11 = res1$estimate[1]
      p12 = res1$estimate[2]
      V11 = res1$var[1,1]
      V12 = res1$var[2,2]
      CV1 = res1$var[1,2]
      S1 <- function(x) {1-(1+x^(-p11))^(-p12)}
      dS1.p1 <- function(x) {-p12*x^(-p11)*(1+x^(-p11))^(-p12-1)*log(x)}
      dS1.p2 <- function(x) {(1+x^(-p11))^(-p12)*log(1+x^(-p11))}
    }
  )

  switch (s2,
    "exponential" = {
      p21 = res2$estimate[1]
      p22 = 0
      V21 = res2$var[1]
      V22 = 0
      CV2 = 0
      S2 <- function(x) {exp(-p21*x)}
      dS2.p1 <- function(x) {-x*exp(-p21*x)}
      dS2.p2 <- function(x) {0}
      dS2.t <- function(x) {-p21*exp(-p21*x)}
      dS2.t.p1 <- function(x) {(p21*x-1)*exp(-p21*x)}
      dS2.t.p2 <- function(x) {0}
    },
    "weibull" = {
      p21 = res2$estimate[1]
      p22 = res2$estimate[2]
      V21 = res2$var[1,1]
      V22 = res2$var[2,2]
      CV2 = res2$var[1,2]
      S2 <- function(x) {exp(-p21*x^p22)}
      dS2.p1 <- function(x) {-x^p22*exp(-p21*x^p22)}
      dS2.p2 <- function(x) {-p21*x^p22*log(x)*exp(-p21*x^p22)}
      dS2.t <- function(x) {-p21*p22*x^(p22-1)*exp(-p21*x^p22)}
      dS2.t.p1 <- function(x) {(p21*x^p22-1)*p22*x^(p22-1)*exp(-p21*x^p22)}
      dS2.t.p2 <- function(x) {(p21*p22*log(x)*x^p22-p22*log(x)-1)*p21*x^(p22-1)*exp(-p21*x^p22)}
    },
    "gamma" = {
      p21 = res2$estimate[1]
      p22 = res2$estimate[2]
      V21 = res2$var[1,1]
      V22 = res2$var[2,2]
      CV2 = res2$var[1,2]
      S2 <- function(x) {1-pgamma(x, shape = p21, scale = 1/p22)}
      dS2.p1 <- function(x) {
        h = 1e-5
        ((1-pgamma(x, shape = p21+h, rate = p22)) - (1-pgamma(x, shape = p21, rate = p22))) / h
      }
      dS2.p2 <- function(x) {-(p22*x)^(p21-1)*exp(-p22*x)*x/gamma(p21)}
      dS2.t <- function(x) {-(p22*x)^(p21-1)*exp(-p22*x)*p22/gamma(p21)}
      dS2.t.p1 <- function(x) {(p22*x)^(p21-1)*exp(-p22*x)*p22*(-log(p22*x)+digamma(p21))/gamma(p21)}
      dS2.t.p2 <- function(x) {-x^(p21-1)*p22^(p21-1)*exp(-p22*x)*(p21-p22*x)/gamma(p21)}
    },
    "log-normal" = {
      p21 = res2$estimate[1]
      p22 = res2$estimate[2]
      V21 = res2$var[1,1]
      V22 = res2$var[2,2]
      CV2 = res2$var[1,2]
      S2 <- function(x) {1-plnorm(x, meanlog = p21, sdlog = p22)}
      dS2.p1 <- function(x) {
        h = 1e-5
        ((1-plnorm(x, meanlog = p21+h, sdlog = p22)) - (1-plnorm(x, meanlog = p21, sdlog = p22))) / h
      }
      dS2.p2 <- function(x) {
        h = 1e-5
        ((1-plnorm(x, meanlog = p21, sdlog = p22+h)) - (1-plnorm(x, meanlog = p21, sdlog = p22))) / h
      }
      dS2.t <- function(x) {-1/(x*sqrt(2*pi)*p22)*exp(-1/(2*(p22^2))*(log(x)-p21)^2)}
      dS2.t.p1 <- function(x) {-1/(x*sqrt(2*pi)*p22^3)*(log(x)-p21)*exp(-(log(x)-p21)^2/(2*(p22^2)))}
      dS2.t.p2 <- function(x) {-1/(x*sqrt(2*pi)*p22^2)*exp(-(log(x)-p21)^2/(2*(p22^2)))*((log(x)-p21)^2/(p22^2)-1)}
    },
    "burr3" = {
      p21 = res2$estimate[1]
      p22 = res2$estimate[2]
      V21 = res2$var[1,1]
      V22 = res2$var[2,2]
      CV2 = res2$var[1,2]
      S2 <- function(x) {1-(1+x^(-p21))^(-p22)}
      dS2.p1 <- function(x) {-p22*x^(-p21)*(1+x^(-p21))^(-p22-1)*log(x)}
      dS2.p2 <- function(x) {(1+x^(-p21))^(-p22)*log(1+x^(-p21))}
      dS2.t <- function(x) {-p21*p22*x^(-p21-1)*(1+x^(-p21))^(-p22-1)}
      dS2.t.p1 <- function(x) {-p22*x^(-p21-1)*(1+x^(-p21))^(-p22-1)*(1 - p21*log(x)+p21*(p22+1)*(1+x^(-p21))^(-1)*x^(-p21)*log(x))}
      dS2.t.p2 <- function(x) {-p21*x^(-p21-1)*(1+x^(-p21))^(-p22-1)*(1-p22*log(1+x^(-p21)))}
    }
  )

  switch (copula,
    "clayton" = {
      f.dC.p11 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = S1(t)^(-theta)+S2(t)^(-theta)-1
        res = -dS2.t(t) * S1(t)^(-theta-1) * S2(t)^(-theta-1) * (theta+1) * dS1.p1(t) * A^(-1/theta-2) * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res)| is.infinite(res),0,res)
      }

      f.dC.p12 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = S1(t)^(-theta)+S2(t)^(-theta)-1
        res = -dS2.t(t)*S1(t)^(-theta-1)*S2(t)^(-theta-1)*(theta+1)*dS1.p2(t)*A^(-1/theta-2) * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res)| is.infinite(res), 0, res)
      }

      f.dC.p21 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = S1(t)^(-theta)+S2(t)^(-theta)-1
        B = dS2.t.p1(t)*S2(t)^(-theta-1)*A^(-1/theta-1)
        C = -(theta+1)*dS2.t(t)*dS2.p1(t)*S2(t)^(-theta-2)*A^(-1/theta-1)
        D = (theta+1)*dS2.t(t)*dS2.p1(t)*S2(t)^(-2*theta-2)*A^(-1/theta-2)
        res = -(B+C+D) * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res)| is.infinite(res), 0, res)
      }

      f.dC.p22 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = S1(t)^(-theta)+S2(t)^(-theta)-1
        B = dS2.t.p2(t)*S2(t)^(-theta-1)*A^(-1/theta-1)
        C = -(theta+1)*dS2.t(t)*dS2.p2(t)*S2(t)^(-theta-2)*A^(-1/theta-1)
        D = (theta+1)*dS2.t(t)*dS2.p2(t)*S2(t)^(-2*theta-2)*A^(-1/theta-2)
        res = -(B+C+D) * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res)| is.infinite(res), 0, res)
      }

      kendall = copula.param/(copula.param+2)
    },
    "gumbel" = {
      f.dC.p11 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = (-log(S1(t)))^(theta+1) + (-log(S2(t)))^(theta+1)
        res = -dS2.t(t) * (-log(S1(t)))^theta * (-log(S2(t)))^theta * exp(-A^(1 / (theta + 1))) * dS1.p1(t) / (S1(t) * S2(t)) * (A^(-2*theta/(1 + theta)) + theta * A^((-2*theta - 1) / (1 + theta))) * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res), 0, res)
      }

      f.dC.p12 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = (-log(S1(t)))^(theta+1) + (-log(S2(t)))^(theta+1)
        res = -dS2.t(t) * (-log(S1(t)))^theta * (-log(S2(t)))^theta * exp(-A^(1 / (theta + 1))) * dS1.p2(t) / (S1(t) * S2(t)) * (A^(-2*theta/(1 + theta)) + theta * A^((-2*theta - 1) / (1 + theta))) * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res),0,res)
      }

      f.dC.p21 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = (-log(S1(t)))^(theta+1) + (-log(S2(t)))^(theta+1)
        B = dS2.t.p1(t)
        C = dS2.t(t) * dS2.p1(t) * A^(-theta / (theta + 1)) * (-log(S2(t)))^theta / S2(t)
        D = theta * dS2.t(t) * dS2.p1(t) * A^((-theta-1) / (theta + 1)) * (-log(S2(t)))^theta / S2(t)
        E = -theta * dS2.t(t) * dS2.p1(t) * (-log(S2(t)))^(-1) / S2(t)
        G = -dS2.t(t) * dS2.p1(t) / S2(t)
        res = -exp(-A^(1 / (theta + 1))) * A^(-theta / (theta + 1)) * (-log(S2(t)))^theta / S2(t) * (B + C + D + E + G) * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res), 0, res)
      }

      f.dC.p22 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = (-log(S1(t)))^(theta+1) + (-log(S2(t)))^(theta+1)
        B = dS2.t.p2(t)
        C = dS2.t(t) * dS2.p2(t) * A^(-theta / (theta + 1)) * (-log(S2(t)))^theta / S2(t)
        D = theta * dS2.t(t) * dS2.p2(t) * A^((-theta-1) / (theta + 1)) * (-log(S2(t)))^theta / S2(t)
        E = -theta * dS2.t(t) * dS2.p2(t) * (-log(S2(t)))^(-1) / S2(t)
        G = -dS2.t(t) * dS2.p2(t) / S2(t)
        res = -exp(-A^(1 / (theta + 1))) * A^(-theta / (theta + 1)) * (-log(S2(t)))^theta / S2(t) * (B + C + D + E + G) * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res), 0, res)
      }

      kendall = copula.param/(copula.param+1)
    },
    "frank" = {
      f.dC.p11 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = (exp(-theta)-1)+(exp(-theta*S1(t))-1)*(exp(-theta*S2(t))-1)
        res = theta * dS2.t(t) * exp(-theta*S2(t)) * exp(-theta*S1(t)) * dS1.p1(t) * (exp(-theta)-1) / A^2 * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res), 0, res)
      }

      f.dC.p12 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = (exp(-theta)-1)+(exp(-theta*S1(t))-1)*(exp(-theta*S2(t))-1)
        res = theta * dS2.t(t) * exp(-theta*S2(t)) * exp(-theta*S1(t)) * dS1.p2(t) * (exp(-theta)-1) / A^2 * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res), 0, res)
      }

      f.dC.p21 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = (exp(-theta)-1)+(exp(-theta*S1(t))-1)*(exp(-theta*S2(t))-1)
        B = dS2.t.p1(t) * exp(-theta*S2(t)) / A
        C = - theta * dS2.t(t) * exp(-theta*S2(t)) * dS2.p1(t) * (exp(-theta) - exp(-theta*S1(t))) / A^2
        res = - (exp(-theta*S1(t))-1) * (B + C) * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res), 0, res)
      }

      f.dC.p22 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = (exp(-theta)-1)+(exp(-theta*S1(t))-1)*(exp(-theta*S2(t))-1)
        B = dS2.t.p2(t) * exp(-theta*S2(t)) / A
        C = - theta * dS2.t(t) * exp(-theta*S2(t)) * dS2.p2(t) * (exp(-theta) - exp(-theta*S1(t))) / A^2
        res = - (exp(-theta*S1(t))-1) * (B + C) * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res), 0, res)
      }

      kendall = 1+4/copula.param*(integrate(function(y) y/(exp(y)-1), 0, copula.param)$value/copula.param-1)
    },
    "fgm" = {
      f.dC.p11 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        res = - dS2.t(t) * (dS1.p1(t) + theta * dS1.p1(t) * (1 - 2 * S1(t)) * (1 - 2 * S2(t))) * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res), 0, res)
      }

      f.dC.p12 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        res = - dS2.t(t) * (dS1.p2(t) + theta * dS1.p2(t) * (1 - 2 * S1(t)) * (1 - 2 * S2(t))) * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res),0,res)
      }

      f.dC.p21 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = dS2.t.p1(t) * (S1(t) + theta * S1(t) * (1 - S1(t)) * (1 - 2 * S2(t)))
        B = -dS2.t(t) * theta * S1(t) * (1 - S1(t)) * 2 * dS2.p1(t)
        res = -(A + B) * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res), 0, res)
      }

      f.dC.p22 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = dS2.t.p2(t) * (S1(t) + theta * S1(t) * (1 - S1(t)) * (1 - 2 * S2(t)))
        B = -dS2.t(t) * theta * S1(t) * (1 - S1(t)) * 2 * dS2.p2(t)
        res = -(A + B) * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res), 0, res)
      }

      kendall = 2/9*copula.param
    },
    "gb" = {
      f.dC.p11 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = (1 - theta * log(S1(t))) * S2(t)^(-theta * log(S1(t)))
        B = -theta * S2(t)^(-theta * log(S1(t)))
        C = -theta * (1 - theta * log(S1(t))) * S2(t)^(-theta * log(S1(t))) * log(S2(t))
        res = -dS2.t(t) * dS1.p1(t) * (A+B+C) * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res), 0, res)
      }

      f.dC.p12 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = (1 - theta * log(S1(t))) * S2(t)^(-theta * log(S1(t)))
        B = -theta * S2(t)^(-theta * log(S1(t)))
        C = -theta * (1 - theta * log(S1(t))) * S2(t)^(-theta * log(S1(t))) * log(S2(t))
        res = -dS2.t(t) * dS1.p2(t) * (A+B+C) * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res), 0, res)
      }

      f.dC.p21 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = dS2.t.p1(t) * S2(t)^(-theta * log(S1(t))) - dS2.t(t) * S2(t)^(-theta * log(S1(t)) - 1) * theta * log(S1(t)) * dS2.p1(t)
        res = -S1(t) * (1 - theta * log(S1(t))) * A * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res), 0, res)
      }

      f.dC.p22 <- function(x, theta){
        t = exp(pi/2 * sinh(x))
        A = dS2.t.p2(t) * S2(t)^(-theta * log(S1(t))) - dS2.t(t) * S2(t)^(-theta * log(S1(t)) - 1) * theta * log(S1(t)) * dS2.p2(t)
        res = -S1(t) * (1 - theta * log(S1(t))) * A * pi/2 * cosh(x) * exp(pi/2 * sinh(x))
        ifelse(is.nan(res), 0, res)
      }

      kendall = 1-4/copula.param*(integrate(function(y) y*(1-copula.param*log(y))*log(1-copula.param*log(y)), 0, 1)$value)
    },
  )



  dC.p11 <- integrate(f.dC.p11, -Inf, Inf, theta = copula.param, abs.tol=1e-25)$value
  dC.p12 <- integrate(f.dC.p12, -Inf, Inf, theta = copula.param, abs.tol=1e-25)$value
  dC.p21 <- integrate(f.dC.p21, -Inf, Inf, theta = copula.param, abs.tol=1e-25, subdivisions = 2000L)$value
  dC.p22 <- integrate(f.dC.p22, -Inf, Inf, theta = copula.param, abs.tol=1e-25, subdivisions = 2000L)$value

  var.a = dC.p11^2*V11+dC.p12^2*V12+dC.p21^2*V21+dC.p22^2*V22+2*dC.p11*dC.p12*CV1+2*dC.p21*dC.p22*CV2

  p.hat = MW.comp(copula = copula, copula.param = copula.param, s1 = s1, s2 = s2, S1.param = c(p11, p12), S2.param = c(p21, p22))

  if(logit){
    logit.hat = log(p.hat/(1-p.hat))
    var.la = var.a/(p.hat*(1-p.hat))^2
    SE = sqrt(var.la)
    CI.l_u = logit.hat + qnorm(1-(alpha/2))*SE
    CI.l_l = logit.hat - qnorm(1-(alpha/2))*SE
    lower.l = exp(CI.l_l)/(1+exp(CI.l_l))
    upper.l = exp(CI.l_u)/(1+exp(CI.l_u))
    CI = c(lower.l, upper.l)
    estimate = p.hat
    pval = 2*pnorm((logit.hat-0)/SE, lower.tail = FALSE)
  }
  else{
    SE = sqrt(var.a)
    CI_u = p.hat + qnorm(1-(alpha/2))*SE
    CI_l = p.hat - qnorm(1-(alpha/2))*SE
    CI = c(CI_l, CI_u)
    estimate = p.hat
    pval = 2*pnorm((p.hat-0.5)/SE, lower.tail = FALSE)
  }

  return(list(parameter = c(p11, p12, p21, p22), estimate = estimate, SE = SE, CI = CI, P.value = pval, kendall = kendall, logit.transformed = logit))
}


#' @title Plot method for Mann-Whitney effect under parametric copula models
#' @description \code{MW.plot} plots the results, the parametric estimator and their confidence intervals, for the Mann-Whitney effect under parametric survival functions and copulas.
#'
#' @usage MW.plot(t.event, event, group,
#' copula = c("clayton", "gumbel", "frank", "fgm", "gb"),
#' lower = 0, upper = 1,
#' s1 = c("exponential", "weibull", "gamma", "log-normal", "burr3"),
#' s2 = c("exponential", "weibull", "gamma", "log-normal", "burr3"),
#' par1 = c(0, 0),
#' par2 = c(0, 0),
#' alpha = 0.05,
#' logit = FALSE,
#' xaxis = 2
#' )
#'
#' @param t.event a vector for time-to-event.
#' @param event a vector for event indicator.
#' @param group a vector for group indicator.
#' @param copula copula family.
#' @param lower the lower end points of the interval of the copula parameter.
#' @param upper the lower end points of the interval of the copula parameter.
#' @param s1 a parametric survival function for S1. Available options include; "exponential", "weibull", "gamma", "log-normal", "burr3".
#' @param s2 a parametric survival function for S2. Available options include; "exponential", "weibull", "gamma", "log-normal", "burr3".
#' @param par1 initial value(s) for the parameters of S1.
#' @param par2 initial value(s) for the parameters of S2.
#' @param alpha significance level.
#' @param logit logical; if TRUE, the estimator and CI is logit-transformed.
#' @param xaxis a indicator specifying whether the xaxis is the copula parameter (xaxis = 1) or Kendall's tau (xaxis = 2).
#'
#' @return No return value, called for side effects (generates a plot).
#'
#' @examples
#' ##Exponential distributions
#' #set distribution parameter
#' lambda1 = 1
#' lambda2 = 2
#'
#' #generate time to event
#' u = runif(100)
#' t.event1 = -log(u) / lambda1
#' t.event1 = sort(t.event1)
#'
#' v = runif(100)
#' t.event2 = -log(v) / lambda2
#' t.event2 = sort(t.event2)
#'
#' #censoring indicator
#' t1c = runif(100, 0, 1.5)
#' t.event1 = (t1c >= t.event1) * t.event1 + (t1c < t.event1) * t1c
#' event1 = 1 * (t1c > t.event1)
#'
#' t2c = runif(100, 0, 0.8)
#' t.event2 = (t2c >= t.event2) * t.event2 + (t2c < t.event2) * t2c
#' event2 = 1 * (t2c > t.event2)
#'
#' t.event = c(t.event1, t.event2)
#' event = c(event1, event2)
#'
#' #group indicator
#' group = rep(c(1, 0), each = 100)
#'
#' MW.plot(t.event, event, group,
#'           copula = "clayton",
#'           lower = 0.2, upper = 0.8,
#'           s1 = "exponential", s2 = "exponential",
#'           par1 = c(0, 0), par2 = c(0, 0), alpha = 0.05, logit = FALSE, xaxis = 2)
#'
#'
#' @importFrom stats integrate
#' @import tidyverse ggplot2 purrr dplyr tidyr
#'
#' @export
#'



MW.plot <- function(t.event, event, group,
                    copula = c("clayton", "gumbel", "frank", "fgm", "gb"),  lower = 0, upper = 1,
                    s1 = c("exponential", "weibull", "gamma", "log-normal", "burr3"), s2 = c("exponential", "weibull", "gamma", "log-normal", "burr3"),
                    par1 = c(0, 0), par2 = c(0, 0), alpha = 0.05, logit = FALSE, xaxis = 2) {

  res = NULL
  param = seq(lower, upper, length.out = 20)
  res = map(param, Vectorize(function(x) MW.Copula(t.event, event, group, copula = copula,
                                                    copula.param = x, s1 = s1, s2 = s2, par1 = par1, par2 = par2, alpha = alpha, logit = logit)[c(2,4,6)], vectorize.args = "x"))

  val <- data.frame(map(res, ~ .x[1])) %>%
    pivot_longer(cols = everything(),
                 values_to = "val")

  CI <- data.frame(t(data.frame(map(res, ~ .x[2]))), param = param) %>%
    rename(lower = "X1", upper = "X2")


  switch (xaxis,
    "1" = {
      dat = bind_cols(val, CI)

      res.plot = ggplot(dat,aes(x=param,y=val)) +
        geom_ribbon(aes(ymin=lower,ymax=upper),alpha=0.2) +
        geom_line() +
        xlab("Copula Parameter") +
        ylab("Probability") +
        geom_hline(yintercept = 0.5, color = "black",linewidth = 1.0) +
        scale_y_continuous(breaks = sort(c(seq(0, 1, by = 0.5), 1.00))) +
        coord_cartesian(ylim=c(0, 1.00)) +
        theme_classic() +
        theme(axis.title = element_text(size = 20),
              axis.text = element_text(size = 12),
              legend.text = element_text(size = 12),
              legend.title = element_text(size = 15))

    },
    "2" = {
      tau <- data.frame(map(res, ~ .x[3])) %>%
        pivot_longer(cols = everything(),
                     values_to = "kendall") %>%
        select(-"name")

      dat = bind_cols(val, CI, tau)

      res.plot = ggplot(dat,aes(x="kendall",y=val)) +
        geom_ribbon(aes(ymin=lower,ymax=upper),alpha=0.2) +
        geom_line() +
        xlab("Kendall's tau") +
        ylab("Probability") +
        geom_hline(yintercept = 0.5, color = "black",linewidth = 1.0) +
        scale_y_continuous(breaks = sort(c(seq(0, 1, by = 0.5), 1.00))) +
        coord_cartesian(ylim=c(0, 1.00)) +
        theme_classic() +
        theme(axis.title = element_text(size = 20),
              axis.text = element_text(size = 12),
              legend.text = element_text(size = 12),
              legend.title = element_text(size = 15))

    }
  )

  res.plot
  return(res.plot)
}
