% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/arriaga.R
\name{sen_arriaga_instantaneous}
\alias{sen_arriaga_instantaneous}
\title{Estimate sensitivity of life expectancy for a set of mortality rates}
\usage{
sen_arriaga_instantaneous(
  mx,
  age = 0:(length(mx1) - 1),
  sex = "t",
  nx = rep(1, length(mx)),
  perturb = 1e-06,
  closeout = TRUE
)
}
\arguments{
\item{mx}{numeric vector of mortality rates (central death rates)}

\item{age}{integer vector of the lower bound of each age group (currently only single ages supported)}

\item{sex}{character Male (\code{"m"}), Female (\code{"f"}), or Total (\code{"t"})}

\item{nx}{integer vector of age intervals, default 1.}

\item{perturb}{numeric constant, a very small number}

\item{closeout}{logical. Default \code{TRUE}. Shall we use the HMD Method Protocol to close out the \code{ax} and \code{qx} values? See details.}
}
\value{
numeric vector of sensitivity of life expectancy to perturbations in \code{mx}.
}
\description{
This implementation gives a good approximation of the sensitivity of life expectancy to perturbations in mortality rates (central death rates). Since the Arriaga approach requires two versions of mortality rates \code{mx1}, \code{mx2}, we create these by slightly perturbing \code{mx} up and down. Then we calculate the Arriaga sensitivity in each direction and take the average. Specifically, we create \code{mx1} and \code{mx2} as
\deqn{m_{x}^{1} = m_x \cdot \left(\frac{1}{1 - h}\right)}
\deqn{m_{x}^{2} = m_x \cdot \left(1 - h\right)}
where \code{h} is small value given by the argument \code{perturb}.
}
\details{
A minor correction might be needed for the final age group for the case of the reverse-direction Arriaga sensitivity. Note also for values of \code{perturb} (h) that are less than \code{1e-7} we might lose stability in results.
}
\examples{
a   <- .001
b   <- .07
x   <- 0:100
mx1 <- a * exp(x * b)
mx2 <- a/2 * exp(x * b)
mx  <- (mx1 + mx2) / 2
s     <- sen_arriaga_instantaneous(mx, age = x)
s1    <- sen_arriaga_instantaneous(mx1, age = x)
s2    <- sen_arriaga_instantaneous(mx2, age = x)
s1_d  <- sen_arriaga(mx1, mx2, age = x)
s2_d  <- sen_arriaga(mx2, mx1, age = x)
delta <- mx2 - mx1
# dots give our point estimate of sensitivity at the average of the rates,
# which is different from the

plot(x,s*delta, ylim = c(0,.3))
lines(x,s1*delta,col = "red")
lines(x,s2*delta,col = "blue")
# the sensitivity of the average is different
# from the average of the sensitivities!
lines(x, ((s1+s2)) / 2 * delta)
# and these are different from the directional sensitivities
# covering the whole space from mx1 to mx2:
lines(x, s1_d*delta, col = "red", lty =2)
lines(x, s2_d*delta, col = "blue", lty =2)
}
