% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim.LTA.R
\name{sim.LTA}
\alias{sim.LTA}
\title{Simulate Data for Latent Transition Analysis (LTA)}
\usage{
sim.LTA(
  N = 500,
  I = 5,
  L = 3,
  distribution = "random",
  times = 2,
  type = "LCA",
  rate = NULL,
  constraint = "VV",
  mean.range = c(-2, 2),
  covs.range = c(0.01, 4),
  poly.value = 5,
  IQ = "random",
  params = NULL,
  is.sort = TRUE,
  covariates = NULL,
  beta = NULL,
  gamma = NULL
)
}
\arguments{
\item{N}{Integer; sample size.}

\item{I}{Integer; number of observed items/indicators per time point.}

\item{L}{Integer; number of latent classes/profiles.}

\item{distribution}{Character; distribution of initial class probabilities when not using covariates or \code{params}.
Options: \code{"uniform"} (equal probabilities) or \code{"random"} (Dirichlet-distributed, default).}

\item{times}{Integer; number of time points (must be \eqn{\geq 1}).}

\item{type}{Character; type of latent model. \code{"LCA"} for categorical indicators (default),
\code{"LPA"} for continuous indicators.}

\item{rate}{List of matrices or NULL; transition probability matrices for non-covariate mode.
Each matrix is \eqn{L \times L} with rows summing to 1. If \code{NULL} (default), matrices are
generated with 0.7 diagonal probability and uniform off-diagonals. \strong{Ignored when \code{times=1}}.}

\item{constraint}{Character; covariance structure for LPA (\code{type="LPA"} only). Options:
\code{"VV"} (unstructured, default), \code{"VE"} (diagonal variance), \code{"EE"} (equal variance).}

\item{mean.range}{Numeric vector; range for randomly generated class means in LPA (default: \code{c(-2, 2)}).}

\item{covs.range}{Numeric vector; range for covariance matrix diagonals in LPA (default: \code{c(0.01, 4)}).}

\item{poly.value}{Integer; number of categories for polytomous LCA items (default: 5).}

\item{IQ}{Character; method for generating item discrimination in LCA. \code{"random"} (default) or fixed values.}

\item{params}{List or NULL; pre-specified parameters for reproducibility (see Details).}

\item{is.sort}{A logical value. If \code{TRUE} (Default), the latent classes will be ordered in descending
order according to \code{P.Z}. All other parameters will be adjusted accordingly
based on the reordered latent classes.}

\item{covariates}{List of matrices or NULL; covariate matrices for each time point. Each matrix must have
dimensions \eqn{N \times p_t} and include an intercept column (first column must be all 1s). If \code{NULL},
covariate mode is disabled. See Details for automatic coefficient generation.}

\item{beta}{Matrix or NULL; initial state regression coefficients of dimension \eqn{p_1 \times L}.
Columns correspond to classes 1 to \eqn{L} (last class \eqn{L} is reference and must be zero).
If \code{NULL} and covariates are used, coefficients are randomly generated from \eqn{\text{Uniform}(-1, 1)}.}

\item{gamma}{List or NULL; transition regression coefficients. Must be a list of length \code{times-1}.
Each element \eqn{t} is a list of length \eqn{L} (previous state). Each sub-list contains \eqn{L} vectors
(next state), where the last vector (reference class) is always \eqn{\mathbf{0}}. \strong{Ignored when \code{times=1}}.
If \code{NULL} and covariates are used with \code{times>=2}, coefficients are randomly generated from
\eqn{\text{Uniform}(-1, 1)} for non-reference classes.}
}
\value{
A list of class \code{"sim.LTA"} containing:
\describe{
\item{\code{responses}}{List of length \code{times}; observed data matrices (\eqn{N \times I}).}
\item{\code{Zs}}{List of length \code{times}; true latent class memberships (\eqn{N \times 1} vectors).}
\item{\code{P.Zs}}{List of length \code{times}; marginal class probabilities at each time.}
\item{\code{par}}{Item parameters for LCA (if \code{type="LCA"}).}
\item{\code{means}}{Class means for LPA (if \code{type="LPA"}).}
\item{\code{covs}}{Class covariance matrices for LPA (if \code{type="LPA"}).}
\item{\code{rate}}{True transition matrices (non-covariate mode only; \code{NULL} when \code{times=1}).}
\item{\code{covariates}}{List of covariate matrices used (covariate mode only).}
\item{\code{beta}}{True initial state coefficients (covariate mode only).}
\item{\code{gamma}}{True transition coefficients (covariate mode only; \code{NULL} when \code{times=1}).}
\item{\code{call}}{Function call.}
\item{\code{arguments}}{Input arguments.}
}
}
\description{
Simulates longitudinal latent class/profile data where initial class membership and transition
probabilities may be influenced by time-varying covariates. Supports both Latent Class Analysis (LCA)
for categorical outcomes and Latent Profile Analysis (LPA) for continuous outcomes. Measurement
invariance is assumed by default (identical item parameters across time).
}
\details{
Covariate Requirements:
\itemize{
\item Covariate matrices must include an intercept (first column = 1). If omitted, the function adds an intercept
and issues a warning.
\item When \code{covariates} is provided but \code{beta} or \code{gamma} is \code{NULL}, coefficients are
randomly generated from \eqn{\text{Uniform}(-1, 1)} (non-reference classes only).
\item The reference class (\eqn{L}) always has zero coefficients (\eqn{\boldsymbol{\beta}_L = \mathbf{0}},
\eqn{\boldsymbol{\gamma}_{l,L} = \mathbf{0}}).
}

Parameter Compatibility:
\itemize{
\item Use \code{params} to fix item parameters (LCA) or class means/covariances (LPA) across simulations.
\item In non-covariate mode, \code{rate} must be a list of \eqn{(times-1)} valid transition matrices (ignored when \code{times=1}).
\item In covariate mode with \code{times>=2}, all three (\code{covariates}, \code{beta}, \code{gamma}) must be consistent in dimensions.
}
}
\section{Model Specification}{

\describe{
\item{Initial Class Probabilities (with covariates):}{
For observation/participant \eqn{n} at time 1, the probability of belonging to latent class \eqn{l} is:
\deqn{P(Z_{n1} = l \mid \mathbf{X}_{n1}) =
      \frac{\exp(\boldsymbol{\beta}_l^\top \mathbf{X}_{n1})}
           {\sum_{k=1}^L \exp(\boldsymbol{\beta}_k^\top \mathbf{X}_{n1})}}
where \eqn{\mathbf{X}_{n1} = (X_{n10}, X_{n11}, \dots, X_{n1M})^\top} is the covariate vector for observation/participant \eqn{n} at time 1,
with \eqn{X_{n10} = 1} (intercept term) and \eqn{X_{n1m}} (\eqn{m=1,\dots,M}) representing the value of the \eqn{m}-th covariate.
The coefficient vector \eqn{\boldsymbol{\beta}_l = (\beta_{l0}, \beta_{l1}, \dots, \beta_{lM})^\top} corresponds element-wise to \eqn{\mathbf{X}_{n1}},
where \eqn{\beta_{l0}} is the intercept and \eqn{\beta_{lm}} (\eqn{m \geq 1}) are regression coefficients for covariates.
Class \eqn{L} is the reference class (\eqn{\boldsymbol{\beta}_L = \mathbf{0}}).
}
\item{Transition Probabilities (with covariates and times>=2):}{
For observation/participant \eqn{n} transitioning from class \eqn{l} at time \eqn{t-1} to class \eqn{k} at time \eqn{t} (\eqn{t \geq 2}):
\deqn{P(Z_{nt} = k \mid Z_{n,t-1} = l, \mathbf{X}_{nt}) =
      \frac{\exp(\boldsymbol{\gamma}_{lkt}^\top \mathbf{X}_{nt})}
           {\sum_{j=1}^L \exp(\boldsymbol{\gamma}_{ljt}^\top \mathbf{X}_{nt})}}
where \eqn{\mathbf{X}_{nt} = (X_{nt0}, X_{nt1}, \dots, X_{ntM})^\top} is the covariate vector at time \eqn{t},
with \eqn{X_{nt0} = 1} (intercept) and \eqn{X_{ntm}} (\eqn{m=1,\dots,M}) as the \eqn{m}-th covariate value.
The coefficient vector \eqn{\boldsymbol{\gamma}_{lkt} = (\gamma_{lkt0}, \gamma_{lkt1}, \dots, \gamma_{lktM})^\top}
corresponds element-wise to \eqn{\mathbf{X}_{nt}}, where \eqn{\gamma_{lkt0}} is the intercept and \eqn{\gamma_{lktm}} (\eqn{m \geq 1})
are regression coefficients. Class \eqn{L} is the reference class (\eqn{\boldsymbol{\gamma}_{lLt} = \mathbf{0}} for all \eqn{l}).
}
\item{Without Covariates or When times=1:}{
Initial probabilities follow a multinomial distribution with probabilities \eqn{\boldsymbol{\pi} = (\pi_1, \dots, \pi_L)}.
When \eqn{times \geq 2}, transitions follow a Markov process with fixed probabilities \eqn{\tau_{lk}^{(t)} = P(Z_t = k \mid Z_{t-1} = l)},
where \eqn{\sum_{k=1}^L \tau_{lk}^{(t)} = 1} for each \eqn{l} and \eqn{t}.
}
}
}

\examples{
####################### Example 1: Single time point (times=1) ######################
library(LCPA)
set.seed(123)
sim_single <- sim.LTA(N = 200, I = 4, L = 3, times = 1, type = "LCA")
print(sim_single)

####################### Example 2: LPA without covariates ######################
set.seed(123)
sim_lta <- sim.LTA(N = 200, I = 3, L = 3, times = 3, type = "LPA", constraint = "VE")
print(sim_lta)

################## Example 3: With custom covariates (times>=2) ######################
set.seed(123)
N <- 200 ## sample size

## Covariates at time point T1
covariates.inter <- rep(1, N) # Intercept term is always 1 for each n
covariates.X1 <- rnorm(N)     # Covariate X1 is a continuous variable
covariates.X2 <- rbinom(N, 1, 0.5) # Covariate X2 is a binary variable
covariates.X1.X2 <- covariates.X1 * covariates.X2 # Interaction between covariates X1 and X2
covariates.T1 <- cbind(inter=covariates.inter, X1=covariates.X1,
                       X2=covariates.X2, X1.X2=covariates.X1.X2) # Combine into covariates at T1

## Covariates at time point T2
covariates.inter <- rep(1, N) # Intercept term is always 1 for each n
covariates.X1 <- rnorm(N)     # Covariate X1 is a continuous variable
covariates.X2 <- rbinom(N, 1, 0.5) # Covariate X2 is a binary variable
covariates.X1.X2 <- covariates.X1 * covariates.X2 # Interaction between covariates X1 and X2
covariates.T2 <- cbind(inter=covariates.inter, X1=covariates.X1,
                       X2=covariates.X2, X1.X2=covariates.X1.X2) # Combine into covariates at T2

covariates <- list(t1=covariates.T1, t2=covariates.T2) # Combine into final covariates list

## Simulate beta coefficients
# 3x3 matrix (last column is zero because the last category is used as reference)
beta <- matrix(c( 0.8, -0.5, 0.0,
                 -0.3, -0.4, 0.0,
                  0.2,  0.8, 0.0,
                 -0.1,  0.2, 0.0), ncol=3, byrow=TRUE)

## Simulate gamma coefficients (only needed when times>=2)
gamma <- list(
  lapply(1:3, function(l) {
    lapply(1:3, function(k) if(k < 3)
           runif(4, -1.0, 1.0) else c(0, 0, 0, 0)) # Last class as reference
  })
)

## Simulate the data
sim_custom <- sim.LTA(
  N=N, I=4, L=3, times=2, type="LPA",
  covariates=covariates,
  beta=beta,
  gamma=gamma
)

summary(sim_custom)

}
