% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get.SE.R
\name{get.SE}
\alias{get.SE}
\title{Compute Standard Errors}
\usage{
get.SE(object, method = "Bootstrap", n.Bootstrap = 100, vis = TRUE)
}
\arguments{
\item{object}{An object of class \code{"LCA"} or \code{"LPA"} returned by \code{\link{LCA}} or \code{\link{LPA}}.}

\item{method}{Character specifying SE calculation method: \code{"Obs"} or \code{"Bootstrap"} (default).}

\item{n.Bootstrap}{Integer. Number of bootstrap replicates when \code{method="Bootstrap"} (default=100).}

\item{vis}{Logical. If \code{TRUE}, displays progress information during estimation (default: \code{TRUE}).}
}
\value{
A list of class \code{"SE"} containing:
\describe{
\item{\code{se}}{Named list of SEs matching parameter structure of input model:
\itemize{
\item LPA: \code{means} (matrix: classes x variables),
\code{covs} (array: vars x vars x classes),
\code{P.Z} (vector: class prob SEs)
\item LCA: \code{par} (array: classes x items x categories),
\code{P.Z} (vector: class prob SEs)
\item Critical Note for \code{"Obs"} method:
Only \emph{free parameters} have non-zero SEs. Non-free parameters (e.g., last class probability in \code{P.Z} due
to sum-to-1 constraint; last category probability in LCA items) have SE=0.
Bootstrap provides SEs for all parameters.
}
}
\item{\code{vcov}}{\code{NULL} for bootstrap. For \code{"Obs"}: variance-covariance matrix (may be regularized). Diagonal
contains squared SEs of free parameters.}
\item{\code{hessian}}{\code{NULL} for bootstrap. For \code{"Obs"}: observed information matrix (pre-regularization).
Dimension = number of free parameters.}
\item{\code{diagnostics}}{Method-specific diagnostics:
\itemize{
\item Bootstrap: \code{n.Bootstrap.requested}, \code{n.Bootstrap.completed}
\item Obs: Hessian computation details, condition number, regularization status, step sizes
}
}
\item{\code{call}}{Function call that generated the object}
\item{\code{arguments}}{List of input arguments}
}
}
\description{
Computes approximate standard errors (SEs) for estimated parameters in Latent Class Analysis (LCA)
or Latent Profile Analysis (LPA) models using two methods:
\itemize{
\item \code{"Bootstrap"}: Non-parametric bootstrap with label-switching correction.
McLachlan & Peel (2004) suggest that 50–100 replicates often provide adequate accuracy
for practical purposes, though more (e.g., 500–1000) may be preferred for publication-quality inference.
\item \code{"Obs"}: Numerical evaluation of the observed information matrix (Hessian of negative log-likelihood)
}
Users should note that \code{\link[LCPA]{get.SE}} computes standard errors based on the observed
information matrix via numerical differentiation, which may lack precision and often yields
ill-conditioned matrices. Therefore, we recommend using \code{method = "Bootstrap"}.
}
\examples{
\donttest{
library(LCPA)
set.seed(123)

# LPA with Bootstrap (minimal replicates for example)
lpa_data <- sim.LPA(N = 500, I = 4, L = 3)
lpa_fit <- LPA(lpa_data$response, L = 3)
se_boot <- get.SE(lpa_fit, method = "Bootstrap", n.Bootstrap = 10)

print(se_boot)
extract(se_boot, "covs")


# LCA with Observed Information (note zeros for constrained parameters)
lca_data <- sim.LCA(N = 500, I = 4, L = 3, poly.value = 5)
lca_fit <- LCA(lca_data$response, L = 3)
se_obs <- get.SE(lca_fit, method = "Obs")

print(se_obs)
extract(se_obs, "par")
}

}
\references{
McLachlan, G. J., & Peel, D. (2004). Finite Mixture Models. Wiley.
https://books.google.com.sg/books?id=c2_fAox0DQoC
}
