\name{residualsHSMM}
\alias{residualsHSMM}
\title{Ordinary Residuals for Hidden Semi-Markov Models}
\description{
Computes ordinary (probability integral transform) residuals for a fitted Hidden Semi-Markov Model (HSMM).
Generates quantile-quantile and autocorrelation plots for assessing model fit via simulation-based envelopes.
}
\usage{
residualsHSMM(x, HSMM, obsdist, dwelldist, M = NA, lag.max = 50, nsim = 100,
              use.theoretical.acf = FALSE, verbose = TRUE, seed = NULL)
}
\arguments{
\item{x}{Numeric vector. The observed data sequence.}
\item{HSMM}{A fitted HSMM object (typically the output from \code{\link{findmleHSMM}}),
containing estimated transition probabilities \code{Pi}, initial probabilities \code{delta},
state-dependent observation parameters \code{observationparameters}, and dwell-time parameters \code{dwellparameters}.}
\item{obsdist}{Character string. Observation distribution. Supported distributions include:
"norm", "pois", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma".}
\item{dwelldist}{Character string. Dwell-time distribution. Supported distributions: "pois", "nbinom", "geom", "betabinom".}
\item{M}{Integer. Maximum dwell time to consider for semi-Markov states. Defaults to \code{min(1000, length(x))} if \code{NA}.}
\item{lag.max}{Integer. Maximum lag for autocorrelation function (ACF) computation. Default is 50.}
\item{nsim}{Integer. Number of simulated datasets to generate for constructing the residual envelope. Default is 100.}
\item{use.theoretical.acf}{Logical. If \code{TRUE}, uses theoretical 95\% ACF bands for white noise instead of simulation-based envelope. Default is \code{FALSE}.}
\item{verbose}{Logical. If \code{TRUE}, prints progress messages during simulation and diagnostic summaries. Default is \code{TRUE}.}
\item{seed}{Integer or \code{NULL}. Random seed for reproducibility. Default is \code{NULL}.}
}
\value{
The function generates QQ and ACF plots and prints messages about residuals outside the 95\% envelope.
An invisible list is also returned containing diagnostic information.
}
\details{
The function computes probability integral transform (PIT) residuals for HSMM observations
by combining forward-backward probabilities with cumulative distribution functions of the
state-dependent observations and dwell-time distributions.
Residuals are assessed via:
\itemize{
\item QQ plots comparing observed residuals to the expected standard normal distribution with a 95\% simulation-based envelope.
\item Autocorrelation function plots to detect remaining temporal dependence with simulation-based or theoretical 95\% bands.
}
This provides a visual check of model adequacy, where ideally residuals behave as independent standard normal variables.
}
\examples{
J <- 3
Pi <- matrix(c(0.0, 0.6, 0.4,
               0.5, 0.0, 0.5,
               0.3, 0.7, 0.0), nrow = J, byrow = TRUE)
obspar <- list(
  mean = c(-2, 0, 3),
  sd   = c(1, 1.5, 2)
)
dwellpar <- list(
  lambda = c(3, 5, 4)
)
# Simulate HSMM data
sim <- generateHSMM(n = 200, J = J, obsdist = "norm",
                    dwelldist = "pois", obspar = obspar,
                    dwellpar = dwellpar, Pi = Pi)
# Fit HSMM using the true parameters
HSMM_true <- findmleHSMM(x = sim$x, J = J, M = 100,
                          obsdist = "norm", dwelldist = "pois",
                          obspar = obspar, dwellpar = dwellpar,
                          Pi = Pi)
# Compute residuals and diagnostic plots
residualsHSMM(x = sim$x, HSMM = HSMM_true,
              obsdist = "norm", dwelldist = "pois",
              M = 100, nsim = 50, lag.max = 30)
}
\seealso{
\code{\link{findmleHSMM}}, for estimating HSMM parameters from data.
\code{\link{generateHSMM}}, for simulating HSMM data.
\code{\link{residualsHMM}}, for residual analysis of hidden Markov models.
}
\author{Aimee Cody}
