\name{globaldecodeHMM}
\alias{globaldecodeHMM}
\title{Global Decoding for Hidden Markov Models}
\description{
Computes the most likely hidden state sequence for a univariate time series
using a fitted Hidden Markov Model (HMM) via global decoding (Viterbi-style).
}
\usage{
globaldecodeHMM(x, HMM, obsdist)
}
\arguments{
\item{x}{Numeric vector. The observed data sequence.}

\item{HMM}{A list containing estimated HMM parameters, typically returned by \code{\link{findmleHMM}}. Must include \code{estimate$Pi} and state-dependent observation parameters.}

\item{obsdist}{Character string. Observation distribution. Supported distributions: "norm", "pois", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma".}
}
\value{
\item{states}{Numeric vector. The most likely hidden state at each observation, computed via global decoding.}
}
\details{
This function implements global decoding for an HMM using the estimated parameters from a fitted model. It calculates the forward probabilities and recursively determines the most probable sequence of hidden states. Global decoding provides the single most likely state sequence, unlike local decoding which gives the most probable state at each time point independently.
}
\examples{
# Example with 3 states, normal observations
J <- 3

# HMM transition matrix
Pi <- matrix(c(0.8, 0.15, 0.05,
               0.1, 0.7, 0.2,
               0.2, 0.3, 0.5), nrow = 3, byrow = TRUE)

# Observation parameters (normal distribution)
obspar <- list(
  mean = c(-2, 0, 3),
  sd = c(0.5, 1, 1.5)
)

# Simulate HMM data
sim_data <- generateHMM(n = 200, J = J, obsdist = "norm", obspar = obspar, Pi = Pi)

# Fit HMM to simulated data
HMM_fit <- findmleHMM(J = J, x = sim_data$x, obsdist = "norm",
                       obspar = obspar, Pi = Pi, EM = FALSE)

# Compute global decoding
decoded_states <- globaldecodeHMM(x = sim_data$x, HMM = HMM_fit, obsdist = "norm")

decoded_states
}
\seealso{
\code{\link{findmleHMM}}, for estimating HMM parameters from data.
\code{\link{generateHMM}}, for simulating HMM data.
\code{\link{localdecodeHMM}}, for local decoding of an HMM.
\code{\link{globaldecodeHSMM}}, for global decoding of a hidden semi-Markov model.
}
\author{Aimee Cody}

