residualsHMM <- function(x, HMM, obsdist, lag.max = 50, verbose = TRUE) {

  # Compute observation CDFs for all states and time points
  obsprobs_cdf_fun <- function(x, HMM, obsdist) {
    n <- length(x)
    J <- length(HMM$estimate$delta)
    cdf <- matrix(NA, nrow = n, ncol = J)

    # Distribution-specific CDF calculations
    if (obsdist == "norm") {
      # Normal CDF
      for (i in 1:n) cdf[i, ] <- pnorm(x[i], mean = HMM$estimate$mean, sd = HMM$estimate$sd)
    }
    if (obsdist == "pois") {
      # Poisson CDF
      for (i in 1:n) cdf[i, ] <- ppois(x[i], lambda = HMM$estimate$lambda)
    }
    if (obsdist == "weibull") {
      # Weibull CDF
      for (i in 1:n) cdf[i, ] <- pweibull(x[i], shape = HMM$estimate$shape, scale = HMM$estimate$scale)
    }
    if (obsdist == "zip") {
      # Zero-inflated Poisson CDF
      for (i in 1:n) {
        for (j in 1:J) {
          lambda_j <- HMM$estimate$lambda[j]
          pi_j <- HMM$estimate$pi[j]
          if (x[i] == 0) {
            cdf[i, j] <- pi_j + (1 - pi_j) * ppois(0, lambda = lambda_j)
          } else {
            cdf[i, j] <- (1 - pi_j) * ppois(x[i], lambda = lambda_j)
          }
        }
      }
    }
    if (obsdist == "nbinom") {
      # Negative binomial CDF
      for (i in 1:n) cdf[i, ] <- pnbinom(x[i], size = HMM$estimate$size, mu = HMM$estimate$mu)
    }
    if (obsdist == "zinb") {
      # Zero-inflated negative binomial CDF
      for (i in 1:n) {
        for (j in 1:J) {
          size_j <- HMM$estimate$size[j]
          mu_j <- HMM$estimate$mu[j]
          pi_j <- HMM$estimate$pi[j]
          if (x[i] == 0) {
            cdf[i, j] <- pi_j + (1 - pi_j) * pnbinom(0, size = size_j, mu = mu_j)
          } else {
            cdf[i, j] <- (1 - pi_j) * pnbinom(x[i], size = size_j, mu = mu_j)
          }
        }
      }
    }
    if (obsdist == "exp") {
      # Exponential CDF
      for (i in 1:n) cdf[i, ] <- pexp(x[i], rate = HMM$estimate$rate)
    }
    if (obsdist == "gamma") {
      # Gamma CDF
      for (i in 1:n) cdf[i, ] <- pgamma(x[i], shape = HMM$estimate$shape, rate = HMM$estimate$rate)
    }
    if (obsdist == "lnorm") {
      # Log-normal CDF
      for (i in 1:n) cdf[i, ] <- plnorm(x[i], meanlog = HMM$estimate$meanlog, sdlog = HMM$estimate$sdlog)
    }
    if (obsdist == "gev") {
      # Generalized extreme value CDF
      for (i in 1:J) {
        cdf[, i] <- pevd(x, loc = HMM$estimate$loc[i],
                         scale = HMM$estimate$scale[i],
                         shape = HMM$estimate$shape[i],
                         log.p = FALSE, type = "GEV")
      }
    }
    if (obsdist == "ZInormal") {
      # Zero-inflated normal CDF
      for (i in 1:n) {
        for (j in 1:J) {
          mean_j <- HMM$estimate$mean[j]
          sd_j <- HMM$estimate$sd[j]
          pi_j <- HMM$estimate$pi[j]
          if (x[i] == 0) {
            cdf[i, j] <- pi_j + (1 - pi_j) * pnorm(0, mean = mean_j, sd = sd_j)
          } else {
            cdf[i, j] <- (1 - pi_j) * pnorm(x[i], mean = mean_j, sd = sd_j)
          }
        }
      }
    }
    if (obsdist == "ZIgamma") {
      # Zero-inflated gamma CDF
      for (i in 1:n) {
        for (j in 1:J) {
          shape_j <- HMM$estimate$shape[j]
          rate_j <- HMM$estimate$rate[j]
          pi_j <- HMM$estimate$pi[j]
          if (x[i] == 0) {
            cdf[i, j] <- pi_j + (1 - pi_j) * pgamma(0, shape = shape_j, rate = rate_j)
          } else {
            cdf[i, j] <- (1 - pi_j) * pgamma(x[i], shape = shape_j, rate = rate_j)
          }
        }
      }
    }
    return(cdf)
  }

  n <- length(x)
  J <- length(HMM$estimate$delta)
  Pi <- HMM$estimate$Pi
  delta <- HMM$estimate$delta

  # Compute observation densities
  obsprobs_density <- obsprobs(
    x = x, J = J, obsdist = obsdist,
    obspar = HMM$estimate[1:(length(HMM$estimate) - 2)]
  )

  # Forward algorithm: compute log forward probabilities
  logalpha <- matrix(NA, nrow = n, ncol = J)
  loglik <- 0
  phi <- delta
  for (i in 1:n) {
    if (i > 1) phi <- phi %*% Pi
    phi <- phi * obsprobs_density[i, ]
    sumphi <- sum(phi)
    phi <- phi / sumphi
    loglik <- loglik + log(sumphi)
    logalpha[i, ] <- log(phi) + loglik
  }

  # Backward algorithm: compute log backward probabilities
  logbeta <- matrix(NA, nrow = n, ncol = J)
  phi <- rep(1 / J, J)
  loglik <- log(J)
  logbeta[n, ] <- log(phi) + loglik
  for (i in seq(n - 1, 1, -1)) {
    phi <- Pi %*% (obsprobs_density[i + 1, ] * phi)
    logbeta[i, ] <- log(phi) + loglik
    sumphi <- sum(phi)
    phi <- phi / sumphi
    loglik <- loglik + log(sumphi)
  }

  # Compute observation CDFs
  obsprobs_cdf <- obsprobs_cdf_fun(x, HMM, obsdist)

  # Calculate ordinary residuals using forward-backward probabilities
  probs <- numeric(n)
  for (i in 1:n) {
    # Predictive state probability at time i
    pre <- if (i == 1) {
      delta
    } else {
      la <- logalpha[i - 1, ]
      if (all(la == -Inf)) rep(1 / J, J) else exp(la - mean(la[la != -Inf])) %*% Pi
    }

    # Filtering probability at time i
    lb <- logbeta[i, ]
    post <- if (all(lb == -Inf)) rep(1 / J, J) else exp(lb - mean(lb[lb != -Inf]))

    # Compute residual probability
    denom <- pre %*% post
    probs[i] <- if (denom == 0 || is.na(denom)) NA else
      (pre %*% diag(obsprobs_cdf[i, ]) %*% post) / denom
  }

  # Transform to ordinary residuals via inverse normal CDF
  ordresids <- qnorm(probs)
  if (any(is.na(ordresids))) {
    warning("NA values detected in residuals")
    ordresids <- ordresids[!is.na(ordresids)]
  }

  # Q-Q plot with confidence bands
  qq <- function(res) {
    pord <- function(q, p, n, j) {
      p - porder(q, distn = "norm", mlen = n, mean = 0, sd = 1, j = j, largest = FALSE)
    }
    n <- length(res)
    med <- vector()
    low <- vector()
    upp <- vector()
    mult <- 10
    int.med <- 10 * c(-1, 1)

    # Compute expected order statistics and confidence bands
    for (j in 1:n) {
      check <- 0
      # Find median of j-th order statistic
      while (check == 0) {
        tryCatch({
          med[j] <- uniroot(pord, int.med, p = 0.5, n = n, j = j)$root
          check <- 1
        }, error = function(e) { int.med <- int.med * mult })
      }
      check <- 0
      # Find 2.5% quantile
      while (check == 0) {
        tryCatch({
          low[j] <- uniroot(pord, int.med, p = 0.025, n = n, j = j)$root
          check <- 1
        }, error = function(e) { int.med <- int.med * mult })
      }
      check <- 0
      # Find 97.5% quantile
      while (check == 0) {
        tryCatch({
          upp[j] <- uniroot(pord, int.med, p = 0.975, n = n, j = j)$root
          check <- 1
        }, error = function(e) { int.med <- int.med * mult })
      }
    }
    list(med = med, low = low, upp = upp)
  }

  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar), add = TRUE)

  # Create diagnostic plots
  par(mfrow = c(1, 2))

  # Q-Q plot
  qqres <- qq(ordresids)
  ylimqq <- range(ordresids, qqres$low, qqres$upp)
  plot(qqres$med, sort(ordresids), ylab = "Observed", ylim = ylimqq, xlab = "Expected",
       cex.axis = 1.2, cex.lab = 1.2, pch = 16, cex = 0.5)
  lines(qqres$med, qqres$med, lwd = 0.6)
  lines(qqres$med, qqres$low, col = "red", lty = 2, lwd = 0.6)
  lines(qqres$med, qqres$upp, col = "red", lty = 2, lwd = 0.6)

  # Report proportion outside confidence bands
  propoutside <- mean(sort(ordresids) > qqres$upp | sort(ordresids) < qqres$low)
  if (verbose) {
    message(sprintf("proportion of ordinary residuals outside 95%% interval: (%.1f%%)",
                    100 * propoutside))
  }

  # ACF plot with confidence intervals
  acf.hmm <- acf(ordresids, lag.max = lag.max, plot = FALSE)

  # Compute ACF confidence intervals
  acf.CI <- function(res) {
    n <- length(res)
    n2 <- n - (0:lag.max)
    df <- n2 - 2
    t.low <- qt(0.025, df)
    t.upp <- qt(0.975, df)
    low <- t.low / sqrt(df + t.low^2)
    upp <- t.upp / sqrt(df + t.upp^2)
    list(low = low, upp = upp)
  }

  acf.ci <- acf.CI(ordresids)
  ylimsacf <- range(acf.ci$low, acf.ci$upp, acf.hmm$acf, 1)
  plot(acf.hmm, main = "", ci = 0, ylim = ylimsacf,
       cex.axis = 1.2, cex.lab = 1.2, lwd = 0.6, xlab = "Lag", ylab = "ACF")
  lines(acf.ci$low, lty = 2, col = "blue")
  lines(acf.ci$upp, lty = 2, col = "blue")

  invisible(list(
    residuals = ordresids,
    probabilities = probs,
    qq_bands = qqres,
    acf_values = acf.hmm,
    acf_bands = acf.ci,
    proportion_outside = propoutside
  ))
}


