% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/magev.R
\name{ma.gev}
\alias{ma.gev}
\title{Model Averaging for GEV High Quantile Estimation}
\usage{
ma.gev(
  data = NULL,
  quant = c(0.98, 0.99, 0.995),
  weight = "like1",
  numk = 12,
  B = 200,
  varcom = TRUE,
  trim = 0,
  fig = FALSE,
  bma = FALSE,
  pen = "norm",
  CD = FALSE,
  remle = FALSE
)
}
\arguments{
\item{data}{A numeric vector of data to be fitted (e.g., annual maxima).}

\item{quant}{The probabilities corresponding to high quantiles to be estimated.
Default is c(0.98, 0.99, 0.995).}

\item{weight}{The weighting method name. Options are:
\itemize{
  \item 'like', 'like0', 'like1' (default): Likelihood-based weights (AIC)
  \item 'gLd', 'gLd0', 'gLd1', 'gLd2': Generalized L-moment distance weights
  \item 'med', 'med1', 'med2': Median-based weights
  \item 'cvt': Conventional AIC weights
}
Variants with numbers indicate left trimming level (0, 1, or 2).}

\item{numk}{The number of candidate submodels K. Default is 12.}

\item{B}{The number of bootstrap samples. Default is 200.}

\item{varcom}{Logical. Whether to compute variance of quantile estimates.
Default is TRUE.}

\item{trim}{The number of left trimming for L-moments. Usually 0 (default), 1, or 2.}

\item{fig}{Logical. Whether to produce diagnostic plots. Default is FALSE.}

\item{bma}{Logical. Whether to use Bayesian Model Averaging. Default is FALSE.}

\item{pen}{Penalty type for BMA prior: 'norm' (normal, default) or 'beta'.}

\item{CD}{Logical. Whether to compute Coles-Dixon penalized MLE. Default is FALSE.}

\item{remle}{Logical. Whether to compute restricted MLE. Default is FALSE.}
}
\value{
A list containing:
\itemize{
  \item mle.hosking - MLE estimates in Hosking style (mu, sigma, xi)
  \item qua.mle - Quantile estimates from MLE
  \item mle.cov3 - Covariance matrix of MLE (3x3)
  \item qua.se.mle.delta - Standard errors of MLE quantiles (delta method)
  \item lme - L-moment estimates (mu, sigma, xi)
  \item lme.cov3 - Covariance matrix of LME (bootstrap)
  \item qua.lme - Quantile estimates from LME
  \item qua.se.lme.boots - Standard errors of LME quantiles (bootstrap)
  \item qua.ma - Model-averaged quantile estimates
  \item w.ma - Weights used for model averaging
  \item fixw.se.ma - Asymptotic SE under fixed weights
  \item ranw.se.ma - Asymptotic SE under random weights
  \item surr - Surrogate model parameters (mu, sigma, xi)
  \item pick_xi - Selected xi values for K submodels
  \item qua.bma - (if bma=TRUE) BMA quantile estimates
  \item w.bma - (if bma=TRUE) BMA weights
  \item mle.CD - (if CD=TRUE) Coles-Dixon penalized MLE
  \item qua.CD - (if CD=TRUE) Quantile estimates from CD-MLE
  \item remle1 - (if remle=TRUE) Restricted MLE (first constraint)
  \item qua.remle1 - (if remle=TRUE) Quantile estimates from remle1
  \item remle2 - (if remle=TRUE) Restricted MLE (second constraint)
  \item qua.remle2 - (if remle=TRUE) Quantile estimates from remle2
  \item quant - The quantile probabilities used
}
}
\description{
This function estimates high quantiles of the Generalized Extreme Value (GEV)
distribution using model averaging with mixed criteria. It combines Maximum
Likelihood Estimation (MLE) and L-moment Estimation (LME) to construct
candidate submodels and assign weights effectively.
}
\details{
The model averaging approach works as follows:
\enumerate{
  \item MLE and LME of GEV parameters are computed
  \item K candidate shape parameters (xi) are selected from profile likelihood CI
  \item For each candidate xi, MLE with fixed xi is computed
  \item Weights are assigned based on the selected method
  \item Final quantile estimates are weighted averages across submodels
}

The weighting schemes include:
\itemize{
  \item 'like': AIC-based weights using likelihood with fixed xi
  \item 'gLd': Weights based on generalized L-moment distance
  \item 'med': Weights based on median and L-moment distance
  \item 'cvt': Conventional AIC weights
}

When bma=TRUE, Bayesian model averaging is applied with prior specified by pen.
}
\examples{
# Load example data
data(streamflow)
x <- streamflow$r1

# Basic usage with likelihood weights
result <- ma.gev(x, quant = c(0.95, 0.99), weight = 'like1', B = 100)
print(result$qua.ma)     # Model-averaged quantiles
print(result$qua.mle)    # MLE quantiles for comparison
print(result$qua.lme)    # LME quantiles for comparison

# Using generalized L-moment distance weights
result2 <- ma.gev(x, quant = c(0.95, 0.99), weight = 'gLd', B = 100)
print(result2$w.ma)      # Model weights

}
\references{
Shin, Y., Shin, Y., & Park, J. S. (2026). Model averaging with mixed criteria
for estimating high quantiles of extreme values: Application to heavy rainfall.
\emph{Stochastic Environmental Research and Risk Assessment}, 40(2), 47.
\doi{10.1007/s00477-025-03167-x}
}
\seealso{
\code{\link{glme.gev}} for stationary GLME estimation,
  \code{\link{magev.ksensplot}} for K sensitivity analysis,
  \code{\link{magev.qqplot}} for Q-Q diagnostic plots,
  \code{\link{magev.rlplot}} for return level plots.
}
\author{
Yonggwan Shin, Seokkap Ko, Jihong Park, Yire Shin, Jeong-Soo Park
}
