% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_pred.R
\name{predicted_plot3d}
\alias{predicted_plot3d}
\title{Plot a LAS point cloud in 3D colored by a class field}
\usage{
predicted_plot3d(
  las,
  field = "Classification",
  bg = "white",
  title = "LAS 3D View",
  class_colors = "default",
  class_labels = "default",
  show_all_classes = NULL,
  downsample = c("none", "voxel", "random"),
  voxel_size = 0.1,
  max_points = 200000L,
  size = 2,
  size_by_height = FALSE,
  size_range = c(0.7, 2.5),
  size_power = 1.2,
  verbose = TRUE,
  zoom = 0.7,
  theta = 0,
  phi = -90
)
}
\arguments{
\item{las}{A \code{LAS} object from \code{lidR}.}

\item{field}{Character. Column name in \code{las@data} used to color points
(e.g., \code{"label"}, \code{"Classification"}).}

\item{bg}{Background color of the 3D scene (passed to \code{rgl::bg3d()}).
Use a dark background (e.g., \code{"black"}) for bright colors, or a light background
(e.g., \code{"white"}) for darker colors.}

\item{title}{Character. Title shown in the rgl window. Non-ASCII characters may be
transliterated for compatibility.}

\item{class_colors}{Controls the class-to-color mapping. Supported forms:
\itemize{
\item \code{"default"}: FuelDeep3D default palette for classes (recommended)
\item \code{"auto"} or \code{NULL}: generate a distinct palette automatically
\item Named character vector: explicit mapping, e.g.
\code{c("0"="#1F77B4","1"="#8B4513","2"="#228B22")}
\item Unnamed character vector: assigned in the order of legend classes
(i.e., \code{show_all_classes} if provided, otherwise \code{sort(unique(field))}),
e.g. \code{c("blue","brown","green")}.
}
Note: color names must be valid R colors (see \code{grDevices::colors()}).
Some CSS names (e.g. \code{"lime"}) are not valid in base R; use hex (e.g. \code{"#00FF00"})
or a valid R name like \code{"limegreen"}.}

\item{class_labels}{Controls the class-to-name mapping (used only in console output).
Supported forms:
\itemize{
\item \code{"default"}: default names for classes 0..3 (printed in the console)
\item \code{"none"} or \code{NULL}: do not rename classes (print raw class values)
\item Named character vector: e.g. \code{c("0"="Ground","1"="Stem","2"="Crown")}
}}

\item{show_all_classes}{Optional. A vector of class values (e.g., \code{c(0,1,2,3)})
to force a stable class order for color assignment and console printing, even if some
classes are not present in the current LAS object.}

\item{downsample}{Downsampling mode:
\itemize{
\item \code{"none"} (default): plot all points
\item \code{"random"}: randomly sample up to \code{max_points}
\item \code{"voxel"}: keep one point per voxel of size \code{voxel_size}
}}

\item{voxel_size}{Numeric. Voxel size (in LAS units) used when \code{downsample = "voxel"}.
Smaller values retain more points; larger values reduce density more aggressively.}

\item{max_points}{Integer. Maximum number of points plotted when \code{downsample = "random"}.}

\item{size}{Numeric. Base point size passed to \code{rgl::points3d()}.
Smaller values show finer detail; larger values increase thickness.}

\item{size_by_height}{Logical. If \code{TRUE}, point thickness increases with height (Z),
implemented by binning points into layers for compatibility across platforms.}

\item{size_range}{Numeric length-2. Multipliers applied to \code{size} when \code{size_by_height = TRUE}.
Example \code{c(0.7, 2.5)} means low points use \code{0.7*size} and high points use \code{2.5*size}.}

\item{size_power}{Numeric. Growth curve for thickness when \code{size_by_height = TRUE}.
Values > 1 emphasize thicker points near the top; values < 1 increase thickness earlier.}

\item{verbose}{Logical. If \code{TRUE}, prints:
\itemize{
\item total points vs plotted points
\item the class-to-color mapping
\item (optional) class display names
}}

\item{zoom, theta, phi}{Camera controls passed to \code{rgl::view3d()}.}
}
\value{
Invisibly returns a list with:
\describe{
\item{mapping}{data.frame of class, name, and color used}
\item{n_total}{number of points in input LAS}
\item{n_plotted}{number of points actually plotted}
\item{downsample}{downsampling mode used}
\item{field}{field used for coloring}
}
}
\description{
Visualize a \code{lidR} \code{LAS} object in an interactive \code{rgl} window and
color points by any discrete class field stored in \code{las@data}.
}
\details{
This function is designed for both \emph{raw} and \emph{predicted} outputs:
\itemize{
\item \code{field = "label"}: color by original labels stored in \code{las@data$label}
\item \code{field = "Classification"}: color by predicted classes stored in \code{las@data$Classification}
}

A fixed in-window legend overlay is intentionally \strong{not} implemented.
Instead, when \code{verbose = TRUE}, a class-to-color (and optional class-to-name)
mapping is printed to the R console for clarity and reproducibility.
}
\examples{
if (requireNamespace("lidR", quietly = TRUE) && interactive()){

library(lidR)
las <-  readLAS(system.file("extdata", "las", "tree2.laz", package = "FuelDeep3D"))

# 1) Predicted classes (default palette; no legend overlay)
predicted_plot3d(
  las,
  field = "Classification",
  bg = "white",
  title = "Predicted classes"
)

# 2) Raw labels
predicted_plot3d(
  las,
  field = "label",
  bg = "black",
  title = "Original labels"
)

# 3) Named custom colors (stable mapping)
my_cols <- c("0"="#1F77B4", "1"="#8B4513", "2"="#228B22")
my_labs <- c("0"="Ground vegetation", "1"="Leaves/Foliage", "2"="Branch/Stem")
predicted_plot3d(
  las,
  field = "Classification",
  class_colors = my_cols,
  class_labels = my_labs,
  bg = "white"
)

# 4) Unnamed custom colors (assigned in class order)
# If classes are 0,1,2 this maps 0->black, 1->red, 2->green.
predicted_plot3d(
  las,
  field = "Classification",
  show_all_classes = c(0,1,2),
  class_colors = c("black", "red", "#00FF00"), # hex is safest for "lime"
  bg = "white"
)

# 5) Downsample (voxel) for huge point clouds
predicted_plot3d(
  las,
  field = "Classification",
  downsample = "voxel",
  voxel_size = 0.10,
  size = 2,
  bg = "white"
)

# 6) Thickness by height
predicted_plot3d(
  las,
  field = "Classification",
  size = 1.2,
  size_by_height = TRUE,
  size_range = c(0.8, 2.8),
  size_power = 1.2
)
}
}
