% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils.R
\name{plot_3d}
\alias{plot_3d}
\title{Plot a 3D LAS point cloud colored by elevation}
\usage{
plot_3d(
  las,
  bg = "black",
  zlim = NULL,
  height_palette = NULL,
  size = 0.6,
  max_points = 400000L,
  title = "LAS 3D View",
  center = FALSE,
  add_legend = TRUE,
  legend_height_frac = 0.5,
  legend_width_frac = 0.015,
  legend_xpad_frac = 0.03,
  legend_side = "right",
  legend_pos = c(NA_real_, NA_real_),
  legend_label_mode = c("rel_z", "norm_z", "z", "norm"),
  z_digits = 2L,
  z_unit = "m",
  size_by_height = TRUE,
  size_range = c(1, 4),
  size_power = 1.2,
  zoom = 0.7,
  theta = 0,
  phi = -90
)
}
\arguments{
\item{las}{A \code{lidR::LAS} object.}

\item{bg}{Background color for the \pkg{rgl} scene. (e.g., \code{"black"}, \code{"white"}, \code{"#111111"}).}

\item{zlim}{NULL or numeric length-2 vector giving the Z range used for coloring (values are clamped to this range).}

\item{height_palette}{Vector of colors to define the height color ramp
(e.g., \code{c("purple","blue","cyan","yellow","red")}).}

\item{size}{Numeric. Base point size (thickness). If \code{size_by_height = TRUE}, this acts as a multiplier on \code{size_range}.}

\item{max_points}{Integer. If LAS has more than this many points, a random subsample is plotted for speed.
Use NULL to disable subsampling.}

\item{title}{Character. Plot title (converted to ASCII-safe to avoid \pkg{rgl} text errors).}

\item{center}{Logical. If TRUE, shifts X/Y/Z so minima become 0 (helps visualization when coordinates are large).}

\item{add_legend}{Logical. If TRUE, draws a vertical colorbar and min/max labels.}

\item{legend_height_frac}{Numeric in (0,1]. Visually compress legend height (e.g., 0.5 = half height).}

\item{legend_width_frac}{Numeric. Legend width as a fraction of X-range.}

\item{legend_xpad_frac}{Numeric. Legend x-offset as a fraction of X-range.}

\item{legend_side}{Character. Either \code{"right"} or \code{"left"}.}

\item{legend_pos}{Numeric length-2 vector \code{c(x, y)} to override legend base position (use NA to ignore a coordinate).}

\item{legend_label_mode}{Character. One of \code{"norm_z"}, \code{"z"}, or \code{"norm"}.}

\item{z_digits}{Integer. Number of digits used for legend Z labels.}

\item{z_unit}{Character. Unit label appended to legend Z values (e.g., \code{"m"}). Use \code{""} for none.}

\item{size_by_height}{Logical. If TRUE, point thickness increases with height.}

\item{size_range}{Numeric length-2. Min/max point size (before multiplying by \code{size}) when \code{size_by_height = TRUE}.}

\item{size_power}{Numeric > 0. Controls how quickly thickness increases with height (larger = more emphasis at top).}

\item{zoom, theta, phi}{Camera controls passed to \code{rgl::view3d()}.}
}
\value{
Invisibly returns NULL.
}
\description{
Visualize a \code{lidR::LAS} point cloud in 3D using \pkg{rgl}, with points colored by elevation (Z).
Supports subsampling for performance, custom height color ramps, optional coordinate centering,
an optional compact legend, and optional "thickness by height" (points become larger at higher Z).
}
\details{
\strong{Color mapping}
\itemize{
\item Elevations are optionally clamped to \code{zlim} and normalized to \eqn{[0,1]} for palette lookup.
\item When \code{zlim} is provided, values outside the range are clamped so the color scale stays comparable.
}

\strong{Legend}
\itemize{
\item The legend shows the same color scale as used for points.
\item When \code{legend_label_mode = "norm_z"}, labels are shown as
\code{norm=0 (z = ...)} and \code{norm=1 (z = ...)} to clarify that 0/1 refers to the normalized scale.
\item Use \code{legend_height_frac} to shorten the legend visually (e.g., 0.5 = half height).
}

\strong{CRAN / non-interactive environments}
\itemize{
\item During \code{R CMD check}, the function returns early (no \pkg{rgl} window is opened).
\item If \pkg{rgl} is not installed, a warning is raised and the function returns invisibly.
}
}
\examples{
if (requireNamespace("lidR", quietly = TRUE) && interactive()) {
    # Your plot code here
library(lidR)

las <-  readLAS(system.file("extdata", "las", "tree2.laz", package = "FuelDeep3D"))

# 1) Default plot (black bg, legend on, thickness by height)
plot_3d(las)

# 2) Custom palette + white background
plot_3d(
  las,
  bg = "white",
  height_palette = c("purple","blue","cyan","yellow","red"),
  title = "Custom palette"
)

# 3) Fixed Z color scale for comparisons + no legend
plot_3d(
  las,
  zlim = c(0, 40),
  add_legend = FALSE,
  title = "Fixed Z (0-40), no legend"
)

# 4) Turn OFF thickness-by-height; use a single point size
plot_3d(
  las,
  size_by_height = FALSE,
  size = 4,
  title = "Uniform thicker points"
)

# 5) Legend on the LEFT and thicker legend bar
plot_3d(
  las,
  legend_side = "left",
  legend_width_frac = 0.05,
  title = "Legend left"
)

# 6) Make everything thicker (multiplies size_range when size_by_height=TRUE)
plot_3d(
  las,
  size = 1.8,
  size_range = c(1, 7),
  size_power = 1.2,
  title = "Thicker points by height"
)
}

}
