#' Detect inliers based on unusual frequency of ocurrance
#'
#' Returns an index to elements of a numerical vector whose frequency
#' is unusually large relative to most elements, applying the
#' three-sigma edit rule to counts of individual values.  Inliers
#' often represent data values that are incorrect but consistent
#' with the overall data distribution, as in the case of
#' numerically-coded disguised missing data
#'
#' @param x numerical vector in which inliers are to be detected
#' @param t threshold parameter for detecting outlying counts (default value 3)
#'
#' @return index to elements of `x` that occur unusually often, if any
#' @export
#'
#' @examples
#' x <- seq(-1, 1, length = 100)
#' x[45:54] <- 0
#' FindInliers(x)
FindInliers <- function(x, t = 3){
  #
  stopifnot("x must be numeric"= is.numeric(x))
  stopifnot("t must be non-negative"= t >= 0)
  #
  xx <- x[which(!is.na(x))]
  if (length(xx) == 0){
    inIndex <- integer(0)
  } else {
    xTbl <- table(xx)
    xCts <- as.numeric(xTbl)
    inLims <- ComputeOutlierLimits(xCts, "T", t)
    ctIndex <- which(xCts > inLims["upLim"])
    if (length(ctIndex) == 0){
      inIndex <- integer(0)
    } else {
      inIndex <- which(as.character(x) %in% names(xTbl)[ctIndex])
    }
  }
  return(inIndex)
}
